
{*******************************************************}
{                                                       }
{       SecureAnyBox5 Delphi client examples            }
{                                                       }
{       Copyright (c) 2025-26 TDP                       }
{                                                       }
{****************************************************tk*}

program Sab5AddRecordsExample;

{$APPTYPE CONSOLE}

{$R *.res}

{$DEFINE SSL_1_1_1}
{$DEFINE USE_SSL}

uses
  System.Types, System.SysUtils, System.StrUtils, System.Classes, System.IOUtils,
  WinApi.Windows,
{$IFDEF USE_SSL}
 {$IFDEF SSL_1_1_1}
  ssl_openssl11,
  ssl_openssl11_lib,
 {$ELSE}
  ...
 {$ENDIF}
{$ENDIF}
  System.NetEncoding,
  KShield.Classes,
  KShield.Utils,
{$IFDEF CPUX64}
  KShield.x64.Support,
{$ENDIF}
  SAB5.RESTsend,
  SAB5.NaCl,
  SAB5.Client,
  SAB5.DTO,
  SAB5.Types,
  SAB5.Utils;

function KShieldLogin(const AClient: TSab5Client; const AClientName, AClientVersion, AClientBuild, AComputerName: string; out
  LoggedUser: string; out loginStatus: TLoginStatus): Boolean;
var
{$IFDEF CPUX64}
  Status: TKShieldStatus;
{$ELSE}
  KShieldClient: TKShield;
{$ENDIF}
  Info: TKShieldCertInfo;
  LoginCertificate: TMemoryStream;
  EncodedCert: TStringStream;
  Enc: TBase64Encoding;
begin
  Result := False;
  Info.Cn := '';
  LoggedUser := '';

{$IFDEF CPUX64}
  LoginCertificate := TMemoryStream.Create;
{$ELSE}
  KShieldClient := TKShield.Create;
{$ENDIF}
  Enc := TBase64Encoding.Create(0);
  EncodedCert := TStringStream.Create;
  try
{$IFDEF CPUX64}
    if not GetKShieldLoginCertificate64(LoginCertificate, Info, Status) then
      Exit;

    if Status = ksAuthenticated then
    begin
{$ELSE}
    if KShieldClient.Initialized and (KShieldClient.Status = ksAuthenticated) then
    begin
      LoginCertificate := KShieldClient.LoginCertificate;
{$ENDIF}
      if Assigned(LoginCertificate) then
      begin
        Enc.Encode(LoginCertificate, EncodedCert);
        AClient.kshieldCertLogin(EncodedCert.DataString, loginStatus, AClientName, AClientVersion, AClientBuild, AComputerName);

        if AClient.LoggedIn then
        begin
{$IFDEF CPUX64}
          LoggedUser := string(Info.Cn);
{$ELSE}
          if KShieldClient.GetUserInfo(Info) then
            LoggedUser := string(Info.Cn);
{$ENDIF}
          Result := True;
        end;
      end;
    end;
  finally
{$IFDEF CPUX64}
    LoginCertificate.Free;
{$ELSE}
    KShieldClient.Free;
{$ENDIF}
    EncodedCert.Free;
    Enc.Free;
  end;
end;

var
  GroupIdx: Integer = 1;

procedure Example;
label Err;
const
  DefaultURL = 'https://172.22.78.123:8843';
  DefaultUser = 'admin';
  ClientName = 'Stella';
  ClientVersion = '8.88';
  ClientBuild = '999';
var
  Client: TSab5Client;
  User, Pswd, Domain, SecondFactor, AccessCode: string;
  BaseURL: string;
  Records: TRecords;
  SafeBoxes: TSafeBoxes;
  childSafeBoxes: TSafeBoxes;
  Status: TStatus;

  safeBoxGroup: TSafeBox;
  safeBox: TSafeBox;
  safeBoxGroupId: TSabId;
  safeBoxId: TSabId;
  Account, SecAccount, FileRec: TRecord;
  RecordId: TSabId;

  loginStatus: TLoginStatus;
  Logged: Boolean;
  LoggedUser: string;

  FileStream: TFileStream;
  BytesStream: TBytesStream;

{$IFDEF USE_SSL}
  ProgPath: string;
{$ENDIF}
begin
  Records := nil;
  safeBoxGroupId := 0;
  safeBoxId := 0;

{$IFDEF USE_SSL}
  ProgPath := ExtractFilePath(ParamStr(0));
  if not TSab5Client.CheckSSL(ProgPath, True) then
  begin
    writeln('Cannot load ssleay32.dll or libeay32.dll ssl libraries.');
    Exit;
  end;
{$ENDIF}

  Client := TSab5Client.Create;
  try
    try
      Client.ConnectionTimeout := 1000;  // [ms]

      write('URL (', DefaultURL, '): ');
      readln(BaseURL);
      if BaseURL = '' then
        BaseURL := DefaultURL;
      Client.BaseURL := BaseURL;

      if not Client.getStatus(Status) then
      begin
        writeln('getStatus:');
        writeln(Format('SAB Error: %s', [Client.Error.Message]));
        writeln(Format('  Details: #%d %s', [Client.Error.ErrorCode, Client.Error.ErrorKey]));
        Exit;
      end;
      Client.ErrorMessages.Language := sablngEnglish;

      Logged := KShieldLogin(Client, ClientName, ClientVersion, ClientBuild, '', LoggedUser, loginStatus);
      if Logged then
      begin
        writeln;
        if LoggedUser <> '' then
          writeln('Logged via KShield as ', LoggedUser)
        else
          writeln('Logged via KShield');
        writeln;
      end else
      begin
        write('User (', DefaultUser, '): ');
        readln(User);
        if User = '' then
          User := DefaultUser;

        write('Domain: ');
        readln(Domain);
        writeln;
        writeln;

        write('Password: ');
        readln(Pswd);

        writeln(SLineBreak);
      end;

      if Logged or Client.Login(System.StrUtils.IfThen(Domain <> '', Domain + '\' + User, User), Pswd, loginStatus, ClientName,
        ClientVersion, ClientBuild, SAB5.Utils.GetComputerName) then
      begin
        if loginStatus.SecondFactorRequired then
        begin
          write('Second factor: ');
          readln(SecondFactor);

          if not Client.authenticate2F(SecondFactor) then
            Exit;
        end;

        write('Access Code: ');
        readln(AccessCode);
        writeln;

        safeBoxGroup.CreateSafeBoxGroup('ExamSafeBoxGroup-' + IntToStr(GroupIdx), False);
        while not Client.createSafeBox(AccessCode, safeBoxGroup, safeBoxGroupId) do
        begin
          if not ContainsStr(Client.Error.ErrorKey, 'duplicate') then
            goto Err;

          Inc(GroupIdx);
          safeBoxGroup.CreateSafeBoxGroup('ExamSafeBoxGroup-' + IntToStr(GroupIdx), False);
        end;

        safeBox.CreateSafeBox('ExamSafeBox', False);
        if not Client.createChildSafeBox(AccessCode, safeBoxGroupId, safeBox, safeBoxId) then
          goto Err;

        Account.CreateAccount('ExamAccount', 'acc Descr', 'accexam',
          'acc-note', '', '', '', '');
        if Client.createRecord(safeBoxId, AccessCode, Account, RecordId, tbTrue) then
        begin
          if not Client.setPassword(RecordId, AccessCode, 'acc-pswd') then
            goto Err;
        end else
          goto Err;

        SecAccount.CreateSecAccount('ExamSecAccount', 'acc Descr', 'secexam',
          'acc note', '', '', '', '', '', 'sec-note');
        if Client.createRecord(safeBoxId, AccessCode, SecAccount, RecordId, tbTrue) then
        begin
          if not Client.setPassword(RecordId, AccessCode, 'sec-pswd') then
            goto Err;
        end else
          goto Err;

        FileRec.CreateFile('ExamFile', 'file Descr', '', '', 'sec-note');
        if not Client.createRecord(safeBoxId, AccessCode, FileRec, RecordId, tbTrue) then
          goto Err;

        BytesStream := TBytesStream.Create;
        FileStream := TFileStream.Create('FILE.TXT', fmOpenRead + fmShareDenyWrite);
        try
          BytesStream.LoadFromStream(FileStream);
          Client.uploadFile(RecordId, 'FILE.TXT', AccessCode, BytesStream);
        finally
          BytesStream.Free;
          FileStream.Free;
        end;

        if not (Client.Error.Severity in [msNone, msSUCCESS]) then
          goto Err;

        Finalize(FileRec);
        if not Client.getRecord(RecordId, FileRec, AccessCode) then
          goto Err;

        FileStream := TFileStream.Create('downloadedFILE.TXT', fmCreate);
        try
          Client.downloadFile(RecordId, AccessCode, FileStream);
        finally
          FileStream.Free;
        end;

Err:
        if Client.Error.Severity in [msNone, msSUCCESS] then
          writeln('Ok')
        else
          writeln(Format('SAB Error: %s %s', [Client.Error.ErrorKey, Client.Error.Message]));

        Client.Logout;
      end else
        writeln(Format('SAB Error: %s %s', [Client.Error.ErrorKey, Client.Error.Message]));

    except
      on E: Exception do
      begin
        writeln(E.Message, '  (', E.ClassName, ')');
      end;
    end;
  finally
    Client.Free;
  end;
end;

begin
  Example;
end.
