// FileStream.cpp: implementation of the CFileStream class.
//
//////////////////////////////////////////////////////////////////////

#include "StdAfx.h"
#include "utils.h"
#include "filestream.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CFileStream::CFileStream(LPSTR fullname,BOOL fwriteallowed)
{
    m_uiRef = 1;
	
	m_writemode = fwriteallowed;
	m_filehandle = INVALID_HANDLE_VALUE;
	strlcpy(m_fullname,fullname,MAX_PATH-1);
}

CFileStream::~CFileStream()
{
	closefile();
}


STDMETHODIMP CFileStream::QueryInterface (REFIID riid, LPVOID FAR *ppv)
{
    if (IsEqualIID (riid, IID_IUnknown)) 
	{
        *ppv = this;
        m_uiRef++;
        return NOERROR;
    }
    else if (IsEqualIID (riid, IID_IStream)) 
	{
        *ppv = (LPSTREAM) this;
        m_uiRef++;
        return NOERROR;
    }
    else 
	{
        *ppv = NULL;
        return ResultFromScode (E_NOINTERFACE);
    }
}


STDMETHODIMP_(ULONG) CFileStream::AddRef ()
{
    return ++m_uiRef;
}

STDMETHODIMP_(ULONG) CFileStream::Release ()
{
    if (--m_uiRef == 0) {
        delete this;
		return 0;     // Object may be gone!
	}
    return m_uiRef;
}

STDMETHODIMP CFileStream::Read( 
            void __RPC_FAR *pv,
            ULONG cb,
            ULONG __RPC_FAR *pcbRead)
{
	if (m_filehandle == INVALID_HANDLE_VALUE)
		if (!tryopenfile())
			return E_FAIL;
	if (ReadFile(m_filehandle,pv,cb,pcbRead,NULL)) {
		return S_OK;
	} else
		return E_FAIL;
}

STDMETHODIMP CFileStream::Write( 
            /* [size_is][in] */ const void __RPC_FAR *pv,
            /* [in] */ ULONG cb,
            /* [out] */ ULONG __RPC_FAR *pcbWritten)
{
	if (m_filehandle == INVALID_HANDLE_VALUE)
		if (!tryopenfile())
			return E_FAIL;
	if (m_writemode) {
		if (WriteFile(m_filehandle,pv,cb,pcbWritten,NULL))
			return S_OK;
		else
			return E_FAIL;
	}
	return E_NOTIMPL;
}

/*	STREAM_SEEK_SET	= 0,
	STREAM_SEEK_CUR	= 1,
	STREAM_SEEK_END	= 2 

	FILE_BEGIN           0
	FILE_CURRENT         1
	FILE_END             2
*/

STDMETHODIMP CFileStream::Seek( 
            /* [in] */ LARGE_INTEGER dlibMove,
            /* [in] */ DWORD dwOrigin,
            /* [out] */ ULARGE_INTEGER __RPC_FAR *plibNewPosition)
{
	if (m_filehandle == INVALID_HANDLE_VALUE)
		if (!tryopenfile())
			return E_FAIL;
	DWORD mv=(DWORD)dlibMove.QuadPart;
	DWORD retval=SetFilePointer(m_filehandle,mv,NULL,dwOrigin); 
	if (retval==0x0FFFFFFFF) {
		return E_FAIL;
	} else {
		if (plibNewPosition)
			plibNewPosition->QuadPart=retval;
		return S_OK;
	}
}
	
STDMETHODIMP CFileStream::SetSize( 
            /* [in] */ ULARGE_INTEGER libNewSize)
{
	if (m_writemode)
		return S_OK;
	else
		return E_NOTIMPL;
}

#define copytoblocksz 16*1024
        
STDMETHODIMP CFileStream::CopyTo( 
            /* [unique][in] */ IStream __RPC_FAR *pstm,
            /* [in] */ ULARGE_INTEGER cb,
            /* [out] */ ULARGE_INTEGER __RPC_FAR *pcbRead,
            /* [out] */ ULARGE_INTEGER __RPC_FAR *pcbWritten)
{
	char databuf[copytoblocksz];
	ULARGE_INTEGER remainsz,totalread,totalwritten;
	remainsz=cb;

	BOOL remainblocks=remainsz.QuadPart!=0;
	int err=S_OK;
	DWORD thisblocksz,thisblockread,thisblockwritten;
	totalread.QuadPart=0;
	totalwritten.QuadPart=0;
	
	while (remainblocks) {
		thisblocksz=remainsz.QuadPart>copytoblocksz ? copytoblocksz : (DWORD)remainsz.QuadPart;
		err=Read(&databuf,thisblocksz,&thisblockread);
		if SUCCEEDED(err) {
			totalread.QuadPart+=thisblockread;
			remainsz.QuadPart-=thisblocksz;
			err=pstm->Write(&databuf,thisblockread,&thisblockwritten);
			if SUCCEEDED(err)
				totalwritten.QuadPart+=thisblockwritten;
		}
		remainblocks=(remainsz.QuadPart>0 && thisblocksz==thisblockwritten && SUCCEEDED(err));
	}
	if (pcbRead)
		pcbRead->QuadPart=totalread.QuadPart;
	if (pcbWritten)
		pcbWritten->QuadPart=totalwritten.QuadPart;
	return err;
}
        
STDMETHODIMP CFileStream::Commit( 
            /* [in] */ DWORD grfCommitFlags)
{
	return E_NOTIMPL;
}
        
STDMETHODIMP CFileStream::Revert( void)
{
	return E_NOTIMPL;
}
        
STDMETHODIMP CFileStream::LockRegion( 
            /* [in] */ ULARGE_INTEGER libOffset,
            /* [in] */ ULARGE_INTEGER cb,
            /* [in] */ DWORD dwLockType)
{
	return E_NOTIMPL;
}
        
STDMETHODIMP CFileStream::UnlockRegion( 
            /* [in] */ ULARGE_INTEGER libOffset,
            /* [in] */ ULARGE_INTEGER cb,
            /* [in] */ DWORD dwLockType)
{
	return E_NOTIMPL;
}

        
STDMETHODIMP CFileStream::Stat( 
            /* [out] */ STATSTG __RPC_FAR *pstatstg,
            /* [in] */ DWORD grfStatFlag)
{
	if (m_filehandle == INVALID_HANDLE_VALUE)
		if (!tryopenfile())
			return E_FAIL;

	memset(pstatstg,0,sizeof(STATSTG));
	if (grfStatFlag!=STATFLAG_NONAME) {
		pstatstg->pwcsName=(LPOLESTR)CoTaskMemAlloc(strlen(m_fullname)*2+2);
		if (pstatstg->pwcsName)
			MultiByteToWideChar(CP_ACP,0,m_fullname,-1,pstatstg->pwcsName,strlen(m_fullname)+1);
	}
	pstatstg->cbSize.QuadPart=GetFileSize(m_filehandle,NULL);
	pstatstg->clsid=CLSID_NULL;
	if (m_writemode)
		pstatstg->grfMode=GENERIC_READ | GENERIC_WRITE;
	else
		pstatstg->grfMode=GENERIC_READ;
	pstatstg->type=STGTY_STREAM;
	return S_OK;
}

        
STDMETHODIMP CFileStream::Clone( 
            /* [out] */ IStream __RPC_FAR *__RPC_FAR *ppstm)
{
	CFileStream *pFileStream = new CFileStream(m_fullname,m_writemode);
	if (pFileStream) {
		*ppstm=pFileStream;
		return S_OK;
	} else
		return E_FAIL;
}

BOOL CFileStream::tryopenfile()
{
	if (m_writemode)
		m_filehandle = CreateFile (m_fullname,	
                      GENERIC_READ | GENERIC_WRITE,// Open for reading+writing
		      FILE_SHARE_READ,        // Allow reading
                      NULL,                   // No security
                      OPEN_ALWAYS,            // Existing file only
                      FILE_ATTRIBUTE_NORMAL,  // Normal file
                      NULL);                  // No template file
	else
		m_filehandle = CreateFile (m_fullname,	
                      GENERIC_READ,
					      // Open for reading
                      FILE_SHARE_READ,        // Allow reading
                      NULL,                   // No security
                      OPEN_EXISTING,          // Existing file only
                      FILE_ATTRIBUTE_NORMAL,  // Normal file
                      NULL);                  // No template file
	return (m_filehandle != INVALID_HANDLE_VALUE);
}

void CFileStream::closefile()
{
	if (m_filehandle != INVALID_HANDLE_VALUE) {
		CloseHandle(m_filehandle);
		m_filehandle=INVALID_HANDLE_VALUE;
	}
}
