#include <windows.h>
#include <stdio.h>
#include <wchar.h>
#include "wlanfunctions.h"
#include "utils.h"
#include "fsplugin.h"

extern tProgressProc ProgressProc;
extern int PluginNumber;

typedef enum _WLAN_INTERFACE_STATE { 
  wlan_interface_state_not_ready              = 0,
  wlan_interface_state_connected              = 1,
  wlan_interface_state_ad_hoc_network_formed  = 2,
  wlan_interface_state_disconnecting          = 3,
  wlan_interface_state_disconnected           = 4,
  wlan_interface_state_associating            = 5,
  wlan_interface_state_discovering            = 6,
  wlan_interface_state_authenticating         = 7
} WLAN_INTERFACE_STATE, *PWLAN_INTERFACE_STATE;

typedef enum _WLAN_CONNECTION_MODE { 
  wlan_connection_mode_profile,
  wlan_connection_mode_temporary_profile,
  wlan_connection_mode_discovery_secure,
  wlan_connection_mode_discovery_unsecure,
  wlan_connection_mode_auto,
  wlan_connection_mode_invalid
} WLAN_CONNECTION_MODE, *PWLAN_CONNECTION_MODE;

typedef struct _NDIS_OBJECT_HEADER {
  UCHAR  Type;
  UCHAR  Revision;
  USHORT Size;
} NDIS_OBJECT_HEADER, *PNDIS_OBJECT_HEADER;

#define DOT11_SSID_MAX_LENGTH 32

typedef struct _DOT11_SSID {
  ULONG uSSIDLength;
  UCHAR ucSSID[DOT11_SSID_MAX_LENGTH];
} DOT11_SSID, *PDOT11_SSID;

typedef UCHAR DOT11_MAC_ADDRESS[6];
typedef DOT11_MAC_ADDRESS* PDOT11_MAC_ADDRESS;

typedef struct _DOT11_BSSID_LIST {
  NDIS_OBJECT_HEADER Header;
  ULONG              uNumOfEntries;
  ULONG              uTotalNumOfEntries;
  DOT11_MAC_ADDRESS  BSSIDs[1];
} DOT11_BSSID_LIST, *PDOT11_BSSID_LIST;

typedef struct _WLAN_INTERFACE_INFO {
  GUID                 InterfaceGuid;
  WCHAR                strInterfaceDescription[256];
  WLAN_INTERFACE_STATE isState;
} WLAN_INTERFACE_INFO, *PWLAN_INTERFACE_INFO;

typedef struct _WLAN_INTERFACE_INFO_LIST {
  DWORD               dwNumberOfItems;
  DWORD               dwIndex;
  WLAN_INTERFACE_INFO InterfaceInfo[1];
} WLAN_INTERFACE_INFO_LIST, *PWLAN_INTERFACE_INFO_LIST;

typedef enum _DOT11_BSS_TYPE { 
  dot11_BSS_type_infrastructure  = 1,
  dot11_BSS_type_independent     = 2,
  dot11_BSS_type_any             = 3
} DOT11_BSS_TYPE, *PDOT11_BSS_TYPE;

typedef struct _WLAN_CONNECTION_PARAMETERS {
  WLAN_CONNECTION_MODE wlanConnectionMode;
  LPCWSTR              strProfile;
  PDOT11_SSID          pDot11Ssid;
  PDOT11_BSSID_LIST    pDesiredBssidList;
  DOT11_BSS_TYPE       dot11BssType;
  DWORD                dwFlags;
} WLAN_CONNECTION_PARAMETERS, *PWLAN_CONNECTION_PARAMETERS;

typedef DWORD WLAN_REASON_CODE, *PWLAN_REASON_CODE;
typedef ULONG WLAN_SIGNAL_QUALITY, *PWLAN_SIGNAL_QUALITY; 

typedef enum _DOT11_PHY_TYPE {
    dot11_phy_type_unknown = 0,
    dot11_phy_type_any = dot11_phy_type_unknown,
    dot11_phy_type_fhss = 1,
    dot11_phy_type_dsss = 2,
    dot11_phy_type_irbaseband = 3,
    dot11_phy_type_ofdm = 4,
    dot11_phy_type_hrdsss = 5,
    dot11_phy_type_erp = 6,
    dot11_phy_type_ht = 7,
    dot11_phy_type_IHV_start = 0x80000000,
    dot11_phy_type_IHV_end = 0xffffffff
} DOT11_PHY_TYPE, * PDOT11_PHY_TYPE;

typedef enum _DOT11_AUTH_ALGORITHM {
    DOT11_AUTH_ALGO_80211_OPEN = 1,
    DOT11_AUTH_ALGO_80211_SHARED_KEY = 2,
    DOT11_AUTH_ALGO_WPA = 3,
    DOT11_AUTH_ALGO_WPA_PSK = 4,
    DOT11_AUTH_ALGO_WPA_NONE = 5,               // used in NatSTA only
    DOT11_AUTH_ALGO_RSNA = 6,
    DOT11_AUTH_ALGO_RSNA_PSK = 7,
    DOT11_AUTH_ALGO_IHV_START = 0x80000000,
    DOT11_AUTH_ALGO_IHV_END = 0xffffffff
} DOT11_AUTH_ALGORITHM, * PDOT11_AUTH_ALGORITHM;

typedef enum _DOT11_CIPHER_ALGORITHM {
    DOT11_CIPHER_ALGO_NONE = 0x00,
    DOT11_CIPHER_ALGO_WEP40 = 0x01,
    DOT11_CIPHER_ALGO_TKIP = 0x02,
    DOT11_CIPHER_ALGO_CCMP = 0x04,
    DOT11_CIPHER_ALGO_WEP104 = 0x05,
    DOT11_CIPHER_ALGO_WPA_USE_GROUP = 0x100,
    DOT11_CIPHER_ALGO_RSN_USE_GROUP = 0x100,
    DOT11_CIPHER_ALGO_WEP = 0x101,
    DOT11_CIPHER_ALGO_IHV_START = 0x80000000,
    DOT11_CIPHER_ALGO_IHV_END = 0xffffffff
} DOT11_CIPHER_ALGORITHM, * PDOT11_CIPHER_ALGORITHM;

typedef struct DOT11_CIPHER_ALGORITHM_LIST {
    #define DOT11_CIPHER_ALGORITHM_LIST_REVISION_1  1
    NDIS_OBJECT_HEADER Header;
    ULONG uNumOfEntries;
    ULONG uTotalNumOfEntries;
    DOT11_CIPHER_ALGORITHM AlgorithmIds[1];
} DOT11_CIPHER_ALGORITHM_LIST, * PDOT11_CIPHER_ALGORITHM_LIST;

#define WLAN_MAX_PHY_TYPE_NUMBER    8

typedef struct _WLAN_AVAILABLE_NETWORK {
  WCHAR                  strProfileName[256];
  DOT11_SSID             dot11Ssid;
  DOT11_BSS_TYPE         dot11BssType;
  ULONG                  uNumberOfBssids;
  BOOL                   bNetworkConnectable;
  WLAN_REASON_CODE       wlanNotConnectableReason;
  ULONG                  uNumberOfPhyTypes;
  DOT11_PHY_TYPE         dot11PhyTypes[WLAN_MAX_PHY_TYPE_NUMBER];
  BOOL                   bMorePhyTypes;
  WLAN_SIGNAL_QUALITY    wlanSignalQuality;
  BOOL                   bSecurityEnabled;
  DOT11_AUTH_ALGORITHM   dot11DefaultAuthAlgorithm;
  DOT11_CIPHER_ALGORITHM dot11DefaultCipherAlgorithm;
  DWORD                  dwFlags;
  DWORD                  dwReserved;
} WLAN_AVAILABLE_NETWORK, *PWLAN_AVAILABLE_NETWORK;

typedef struct _WLAN_AVAILABLE_NETWORK_LIST {
    DWORD dwNumberOfItems;
    DWORD dwIndex;
    WLAN_AVAILABLE_NETWORK Network[1];
} WLAN_AVAILABLE_NETWORK_LIST, *PWLAN_AVAILABLE_NETWORK_LIST;

typedef struct _WLAN_RAW_DATA {
    // size of the data blob
    DWORD dwDataSize;
    BYTE DataBlob[1];
} WLAN_RAW_DATA, *PWLAN_RAW_DATA;


/********************************************************************************/

typedef unsigned char UINT8;
typedef unsigned char BYTE;

#define GAA_FLAG_SKIP_UNICAST  0x0001
#define GAA_FLAG_SKIP_ANYCAST  0x0002
#define GAA_FLAG_SKIP_MULTICAST  0x0004
#define GAA_FLAG_SKIP_DNS_SERVER  0x0008
#define GAA_FLAG_INCLUDE_PREFIX  0x0010
#define GAA_FLAG_SKIP_FRIENDLY_NAME  0x0020
#define GAA_FLAG_INCLUDE_WINS_INFO  0x0040
#define GAA_FLAG_INCLUDE_GATEWAYS  0x0080
#define GAA_FLAG_INCLUDE_ALL_INTERFACES  0x0100
#define GAA_FLAG_INCLUDE_ALL_COMPARTMENTS  0x0200
#define GAA_FLAG_INCLUDE_TUNNEL_BINDINGORDER  0x0400

typedef struct _SOCKET_ADDRESS {
  LPSOCKADDR lpSockaddr;
  INT        iSockaddrLength;
} SOCKET_ADDRESS, *PSOCKET_ADDRESS;

typedef enum  { 
  IpPrefixOriginOther                = 0,
  IpPrefixOriginManual,
  IpPrefixOriginWellKnown,
  IpPrefixOriginDhcp,
  IpPrefixOriginRouterAdvertisement,
  IpPrefixOriginUnchanged            = 16
} IP_PREFIX_ORIGIN;

typedef enum  { 
  IpSuffixOriginOther             = 0,
  IpSuffixOriginManual,
  IpSuffixOriginWellKnown,
  IpSuffixOriginDhcp,
  IpSuffixOriginLinkLayerAddress,
  IpSuffixOriginRandom,
  IpSuffixOriginUnchanged         = 16
} IP_SUFFIX_ORIGIN;

typedef enum  { 
  IpDadStateInvalid     = 0,
  IpDadStateTentative,
  IpDadStateDuplicate,
  IpDadStateDeprecated,
  IpDadStatePreferred
} IP_DAD_STATE;

typedef struct _IP_ADAPTER_UNICAST_ADDRESS {
  union {
    struct {
      ULONG Length;
      DWORD Flags;
    };
  };
  struct _IP_ADAPTER_UNICAST_ADDRESS  *Next;
  SOCKET_ADDRESS                     Address;
  IP_PREFIX_ORIGIN                   PrefixOrigin;
  IP_SUFFIX_ORIGIN                   SuffixOrigin;
  IP_DAD_STATE                       DadState;
  ULONG                              ValidLifetime;
  ULONG                              PreferredLifetime;
  ULONG                              LeaseLifetime;
  UINT8                              OnLinkPrefixLength;
} IP_ADAPTER_UNICAST_ADDRESS, *PIP_ADAPTER_UNICAST_ADDRESS;

typedef struct _IP_ADAPTER_ANYCAST_ADDRESS {
  union {
    ULONGLONG Alignment;
    struct {
      ULONG Length;
      DWORD Flags;
    };
  };
  struct _IP_ADAPTER_ANYCAST_ADDRESS  *Next;
  SOCKET_ADDRESS                     Address;
} IP_ADAPTER_ANYCAST_ADDRESS, *PIP_ADAPTER_ANYCAST_ADDRESS;

typedef struct _IP_ADAPTER_MULTICAST_ADDRESS {
  union {
    ULONGLONG Alignment;
    struct {
      ULONG Length;
      DWORD Flags;
    };
  };
  struct _IP_ADAPTER_MULTICAST_ADDRESS  *Next;
  SOCKET_ADDRESS                       Address;
} IP_ADAPTER_MULTICAST_ADDRESS, *PIP_ADAPTER_MULTICAST_ADDRESS;

typedef struct _IP_ADAPTER_DNS_SERVER_ADDRESS {
  union {
    ULONGLONG Alignment;
    struct {
      ULONG Length;
      DWORD Reserved;
    };
  };
  struct _IP_ADAPTER_DNS_SERVER_ADDRESS  *Next;
  SOCKET_ADDRESS                        Address;
} IP_ADAPTER_DNS_SERVER_ADDRESS, *PIP_ADAPTER_DNS_SERVER_ADDRESS;

typedef struct _IP_ADAPTER_PREFIX {
  union {
    ULONGLONG  Alignment;
    struct {
      ULONG Length;
      DWORD Flags;
    };
  };
  struct _IP_ADAPTER_PREFIX  *Next;
  SOCKET_ADDRESS            Address;
  ULONG                     PrefixLength;
} IP_ADAPTER_PREFIX, *PIP_ADAPTER_PREFIX;

#define MAX_ADAPTER_ADDRESS_LENGTH 8

typedef enum  { 
  IfOperStatusUp              = 1,
  IfOperStatusDown,
  IfOperStatusTesting,
  IfOperStatusUnknown,
  IfOperStatusDormant,
  IfOperStatusNotPresent,
  IfOperStatusLowerLayerDown
} IF_OPER_STATUS;

typedef struct _IP_ADAPTER_WINS_SERVER_ADDRESS_LH {
    union {
        ULONGLONG Alignment;
        struct {
            ULONG Length;
            DWORD Reserved;
        };
    };
    struct _IP_ADAPTER_WINS_SERVER_ADDRESS_LH *Next;
    SOCKET_ADDRESS Address;
} IP_ADAPTER_WINS_SERVER_ADDRESS_LH, *PIP_ADAPTER_WINS_SERVER_ADDRESS_LH;

typedef struct _IP_ADAPTER_GATEWAY_ADDRESS_LH {
    union {
        ULONGLONG Alignment;
        struct {
            ULONG Length;
            DWORD Reserved;
        };
    };
    struct _IP_ADAPTER_GATEWAY_ADDRESS_LH *Next;
    SOCKET_ADDRESS Address;
} IP_ADAPTER_GATEWAY_ADDRESS_LH, *PIP_ADAPTER_GATEWAY_ADDRESS_LH;

typedef union _NET_LUID {
  ULONG64 Value;
  struct {
    ULONG64 Reserved  :24;
    ULONG64 NetLuidIndex  :24;
    ULONG64 IfType  :16;
  } Info;
} NET_LUID, *PNET_LUID;

typedef NET_LUID IF_LUID, *PIF_LUID;
typedef DWORD NET_IF_COMPARTMENT_ID;
typedef GUID NET_IF_NETWORK_GUID;
typedef int NET_IF_CONNECTION_TYPE;

#define MAX_DHCPV6_DUID_LENGTH  130
#define MAX_DNS_SUFFIX_STRING_LENGTH  256

typedef enum  { 
  TUNNEL_TYPE_NONE     = 0,
  TUNNEL_TYPE_OTHER,
  TUNNEL_TYPE_DIRECT,
  TUNNEL_TYPE_6TO4,
  TUNNEL_TYPE_ISATAP,
  TUNNEL_TYPE_TEREDO,
  TUNNEL_TYPE_IPHTTPS
} TUNNEL_TYPE, *PTUNNEL_TYPE;

typedef struct _IP_ADAPTER_DNS_SUFFIX {
  struct _IP_ADAPTER_DNS_SUFFIX  *Next;
  WCHAR                         String[MAX_DNS_SUFFIX_STRING_LENGTH];
} IP_ADAPTER_DNS_SUFFIX, *PIP_ADAPTER_DNS_SUFFIX;

typedef struct _IP_ADAPTER_ADDRESSES {
  union {
    ULONGLONG Alignment;
    struct {
      ULONG Length;
      DWORD IfIndex;
    };
  };
  struct _IP_ADAPTER_ADDRESSES  *Next;
  PCHAR                              AdapterName;
  PIP_ADAPTER_UNICAST_ADDRESS        FirstUnicastAddress;
  PIP_ADAPTER_ANYCAST_ADDRESS        FirstAnycastAddress;
  PIP_ADAPTER_MULTICAST_ADDRESS      FirstMulticastAddress;
  PIP_ADAPTER_DNS_SERVER_ADDRESS     FirstDnsServerAddress;
  PWCHAR                             DnsSuffix;
  PWCHAR                             Description;
  PWCHAR                             FriendlyName;
  BYTE                               PhysicalAddress[MAX_ADAPTER_ADDRESS_LENGTH];
  DWORD                              PhysicalAddressLength;
  DWORD                              Flags;
  DWORD                              Mtu;
  DWORD                              IfType;
  IF_OPER_STATUS                     OperStatus;
  DWORD                              Ipv6IfIndex;
  DWORD                              ZoneIndices[16];
  PIP_ADAPTER_PREFIX                 FirstPrefix;
  ULONG64                            TransmitLinkSpeed;
  ULONG64                            ReceiveLinkSpeed;
  PIP_ADAPTER_WINS_SERVER_ADDRESS_LH FirstWinsServerAddress;
  PIP_ADAPTER_GATEWAY_ADDRESS_LH     FirstGatewayAddress;
  ULONG                              Ipv4Metric;
  ULONG                              Ipv6Metric;
  IF_LUID                            Luid;
  SOCKET_ADDRESS                     Dhcpv4Server;
  NET_IF_COMPARTMENT_ID              CompartmentId;
  NET_IF_NETWORK_GUID                NetworkGuid;
  NET_IF_CONNECTION_TYPE             ConnectionType;
  TUNNEL_TYPE                        TunnelType;
  SOCKET_ADDRESS                     Dhcpv6Server;
  BYTE                               Dhcpv6ClientDuid[MAX_DHCPV6_DUID_LENGTH];
  ULONG                              Dhcpv6ClientDuidLength;
  ULONG                              Dhcpv6Iaid;
  PIP_ADAPTER_DNS_SUFFIX             FirstDnsSuffix;
} IP_ADAPTER_ADDRESSES, *PIP_ADAPTER_ADDRESSES;

/********************************************************************************/

#define _In_
#define _Reserved_
#define _Out_
#define _In_opt_
#define _Inout_

typedef DWORD (WINAPI *DEF_WlanOpenHandle)(
  _In_        DWORD dwClientVersion,   // 1 XP, 2 Vista
  _Reserved_  PVOID pReserved,
  _Out_       PDWORD pdwNegotiatedVersion,
  _Out_       PHANDLE phClientHandle
);

typedef DWORD (WINAPI *DEF_WlanEnumInterfaces)(
  _In_        HANDLE hClientHandle,
  _Reserved_  PVOID pReserved,
  _Out_       PWLAN_INTERFACE_INFO_LIST *ppInterfaceList
);

typedef DWORD (WINAPI *DEF_WlanCloseHandle)(
  _In_        HANDLE hClientHandle,
  _Reserved_  PVOID pReserved
);

typedef DWORD (WINAPI* DEF_WlanScan)(
  _In_        HANDLE hClientHandle,
  _In_        const GUID *pInterfaceGuid,
  _In_opt_    const PDOT11_SSID pDot11Ssid,
  _In_opt_    const PWLAN_RAW_DATA pIeData,
  _Reserved_  PVOID pReserved
);

typedef VOID (WINAPI *DEF_WlanFreeMemory)(
  _In_  PVOID pMemory
);

typedef DWORD (WINAPI *DEF_WlanGetAvailableNetworkList)(
  _In_        HANDLE hClientHandle,
  _In_        const GUID *pInterfaceGuid,
  _In_        DWORD dwFlags,
  _Reserved_  PVOID pReserved,
  _Out_       PWLAN_AVAILABLE_NETWORK_LIST *ppAvailableNetworkList
);

typedef DWORD (WINAPI *DEF_WlanConnect)(
  _In_        HANDLE hClientHandle,
  _In_        const GUID *pInterfaceGuid,
  _In_        const PWLAN_CONNECTION_PARAMETERS pConnectionParameters,
  _Reserved_  PVOID pReserved
);

typedef DWORD (WINAPI *DEF_WlanSetProfile)(
  _In_        HANDLE hClientHandle,
  _In_        const GUID *pInterfaceGuid,
  _In_        DWORD dwFlags,
  _In_        LPCWSTR strProfileXml,
  _In_opt_    LPCWSTR strAllUserProfileSecurity,
  _In_        BOOL bOverwrite,
  _Reserved_  PVOID pReserved,
  _Out_       DWORD *pdwReasonCode
);

typedef DWORD (WINAPI* DEF_WlanDisconnect)(
  _In_        HANDLE hClientHandle,
  _In_        const GUID *pInterfaceGuid,
  _Reserved_  PVOID pReserved
);

typedef DWORD (WINAPI* DEF_WlanDeleteProfile)(
  _In_        HANDLE hClientHandle,
  _In_        const GUID *pInterfaceGuid,
  _In_        LPCWSTR strProfileName,
  _Reserved_  PVOID pReserved
);

// the following is in Iphlpapi.dll
typedef ULONG (WINAPI *DEF_GetAdaptersAddresses)(
  _In_     ULONG Family,
  _In_     ULONG Flags,
  _In_     PVOID Reserved,
  _Inout_  PIP_ADAPTER_ADDRESSES AdapterAddresses,
  _Inout_  PULONG SizePointer
);

DEF_WlanOpenHandle WlanOpenHandle=NULL;
DEF_WlanEnumInterfaces WlanEnumInterfaces=NULL;
DEF_WlanCloseHandle WlanCloseHandle=NULL;
DEF_WlanScan WlanScan=NULL;
DEF_WlanFreeMemory WlanFreeMemory=NULL;
DEF_WlanGetAvailableNetworkList WlanGetAvailableNetworkList=NULL;
DEF_WlanConnect WlanConnect=NULL;
DEF_WlanSetProfile WlanSetProfile=NULL;
DEF_WlanDisconnect WlanDisconnect=NULL;
DEF_WlanDeleteProfile WlanDeleteProfile=NULL;
// the following is in Iphlpapi.dll
DEF_GetAdaptersAddresses GetAdaptersAddresses=NULL;

BOOL loadtried=false;
HINSTANCE hlib=NULL;
HINSTANCE hlib2=NULL;
HANDLE gClientHandle=NULL;
WCHAR gSsid[80];
DWORD negotiatedversion=0;
GUID FirstInterfaceGUID;

BOOL LoadWlanFunctions()
{
	if (!loadtried) {
		loadtried=true;
		int oldmode=SetErrorMode(0x8001);
		hlib=LoadLibrary(TEXT("wlanapi.dll"));
		hlib2=LoadLibrary(TEXT("iphlpapi.dll"));
		SetErrorMode(oldmode);
		if (hlib && hlib2) {
			WlanOpenHandle=(DEF_WlanOpenHandle)GetProcAddress(hlib,"WlanOpenHandle");
			WlanEnumInterfaces=(DEF_WlanEnumInterfaces)GetProcAddress(hlib,"WlanEnumInterfaces");
			WlanCloseHandle=(DEF_WlanCloseHandle)GetProcAddress(hlib,"WlanCloseHandle");
			WlanScan=(DEF_WlanScan)GetProcAddress(hlib,"WlanScan");
			WlanFreeMemory=(DEF_WlanFreeMemory)GetProcAddress(hlib,"WlanFreeMemory");
			WlanGetAvailableNetworkList=(DEF_WlanGetAvailableNetworkList)GetProcAddress(hlib,"WlanGetAvailableNetworkList");
			WlanConnect=(DEF_WlanConnect)GetProcAddress(hlib,"WlanConnect");
			WlanSetProfile=(DEF_WlanSetProfile)GetProcAddress(hlib,"WlanSetProfile");
			WlanDisconnect=(DEF_WlanDisconnect)GetProcAddress(hlib,"WlanDisconnect");
			WlanDeleteProfile=(DEF_WlanDeleteProfile)GetProcAddress(hlib,"WlanDeleteProfile");
			GetAdaptersAddresses=(DEF_GetAdaptersAddresses)GetProcAddress(hlib2,"GetAdaptersAddresses");
		}
	}
	return (WlanOpenHandle!=NULL && GetAdaptersAddresses!=NULL);
}

int HasWlanNetworkAdapter()
{
	PWLAN_INTERFACE_INFO_LIST pInterfaceList=NULL;
	if (!LoadWlanFunctions())
		return ERROR_WLAN_NOT_OPENED;
	HANDLE clienthandle=NULL;
	DWORD err=WlanOpenHandle(2,NULL,&negotiatedversion,&clienthandle);
	if (err!=ERROR_SUCCESS)   // can be ERROR_SERVICE_NOT_ACTIVE=1062
		return ERROR_WLAN_NOT_OPENED;
	err=WlanEnumInterfaces(clienthandle,NULL,&pInterfaceList);
	BOOL adaptersfound=false;
	if (err==ERROR_SUCCESS) {
		if (pInterfaceList && pInterfaceList->dwNumberOfItems>0) {
			adaptersfound=true;
			FirstInterfaceGUID=pInterfaceList->InterfaceInfo[0].InterfaceGuid;
		}
		if (pInterfaceList)
			WlanFreeMemory(pInterfaceList);
	}
	WlanCloseHandle(clienthandle,NULL);
	return adaptersfound?ERROR_WLAN_OK:ERROR_WLAN_NO_ADAPTERS;
}

WCHAR* xmltemplate=L"<?xml version=\"1.0\" encoding=\"US-ASCII\"?>\
<WLANProfile xmlns=\"http://www.microsoft.com/networking/WLAN/profile/v1\">\n\
    <name>%s</name>\n\
    <SSIDConfig>\n\
        <SSID>\n\
            <name>%s</name>\n\
        </SSID>\n\
    </SSIDConfig>\n\
    <connectionType>ESS</connectionType>\n\
    <connectionMode>auto</connectionMode>\n\
    <autoSwitch>false</autoSwitch>\n\
    <MSM>\n\
        <security>\n\
            <authEncryption>\n\
                <authentication>%s</authentication>\n\
                <encryption>AES</encryption>\n\
                <useOneX>false</useOneX>\n\
            </authEncryption>\n\
			<sharedKey>\n\
				<keyType>passPhrase</keyType>\n\
				<protected>false</protected>\n\
				<keyMaterial>%s</keyMaterial>\n\
			</sharedKey>\n\
        </security>\n\
    </MSM>\n\
</WLANProfile>\n";

#define WLAN_PROFILE_USER 2

#define STATUS_NO_INTERFACE 1
#define STATUS_CONNECTED 2
#define STATUS_CONNECTING 3

DWORD GetConnectionStatus()
{
	if (gClientHandle==0)
		return STATUS_NO_INTERFACE;
	PWLAN_INTERFACE_INFO_LIST pInterfaceList=NULL;
	int err=WlanEnumInterfaces(gClientHandle,NULL,&pInterfaceList);
	BOOL adaptersfound=false;
	if (err==ERROR_SUCCESS) {
		if (pInterfaceList && pInterfaceList->dwNumberOfItems>0) {
			WLAN_INTERFACE_STATE isState=pInterfaceList->InterfaceInfo[0].isState;
			if (isState==wlan_interface_state_connected)
				err=STATUS_CONNECTED;
			else
				err=STATUS_CONNECTING;

		} else
			err=STATUS_NO_INTERFACE;
		if (pInterfaceList)
			WlanFreeMemory(pInterfaceList);
		return err;
	} else
		return STATUS_NO_INTERFACE;
}

int GetWLanWpaType(WCHAR* ssid,HANDLE clienthandle)
{
	int try1=-1;   // 0=WPA2, 1=WPA, -1=SSID not found
	char ansissid[128];
	WideCharToMultiByte(CP_ACP,0,ssid,-1,ansissid,sizeof(ansissid)-1,NULL,NULL);
	PWLAN_AVAILABLE_NETWORK_LIST networklist=NULL;
	int err=WlanGetAvailableNetworkList(clienthandle,&FirstInterfaceGUID,0,NULL,&networklist);
	if (err==0 && networklist!=NULL) {
		int cnt=networklist->dwNumberOfItems;
		for (int i=0;i<cnt;i++) {
			UCHAR* net=networklist->Network[i].dot11Ssid.ucSSID;
			if (networklist->Network[i].dot11Ssid.uSSIDLength>0 && strcmp((char*)net,ansissid)==0) {
				if (networklist->Network[i].bNetworkConnectable &&
					networklist->Network[i].bSecurityEnabled) {
					_DOT11_AUTH_ALGORITHM alg=networklist->Network[i].dot11DefaultAuthAlgorithm;
					if (alg==DOT11_AUTH_ALGO_WPA_PSK)
						try1=1;
					else
						try1=0;
				}
			}
		}
	} else
		try1=-2;
	return try1;
}

DWORD ConnectToSpecificWlan(WCHAR* ssid,WCHAR* passphrase)
{
	if (!LoadWlanFunctions())
		return -1;
	HANDLE clienthandle=NULL;
	DWORD err=WlanOpenHandle(2,NULL,&negotiatedversion,&clienthandle);
	if (err!=ERROR_SUCCESS)   // can be ERROR_SERVICE_NOT_ACTIVE=1062
		return err;
	WCHAR profilexml[2048];

	// first, find the WLAN to find out whether it uses WPA or WPA2!
	int try1=GetWLanWpaType(ssid,clienthandle);
	if (try1==-1) {
		// scan, then try again
		WlanScan(clienthandle,&FirstInterfaceGUID,NULL,NULL,NULL);
		for (int tries=0;tries<5;tries++) {
			Sleep(1000);
			try1=GetWLanWpaType(ssid,clienthandle);
			if (try1>=0)
				break;
		}
	}
	if (try1<0)
		try1=0;

	for (int tries=try1;tries<2;tries++) {
		swprintf(profilexml,xmltemplate,ssid,ssid,tries==0?L"WPA2PSK":L"WPAPSK",passphrase);
		DWORD ReasonCode;
		if (ProgressProc(PluginNumber,TEXT("Creating WLAN Profile..."),TEXT("temp"),0)) {
			err=-1;
			return NULL;
		}
		for (int tries2=0;tries2<2;tries2++) {
			err=WlanSetProfile(
				clienthandle,
				&FirstInterfaceGUID,
				IsVistaOrNewer()?WLAN_PROFILE_USER:0,
				profilexml,
				NULL,
				TRUE,
				NULL,
				&ReasonCode);
			if (err==0)
				break;
			if (tries2==0)   // error 183=profile already exists!
				WlanDeleteProfile(clienthandle,&FirstInterfaceGUID,ssid,NULL);
		}
		if (err!=0) {
			WlanCloseHandle(clienthandle,NULL);
			return err;
		}
		gSsid[0]=0;
		wcsncat(gSsid,ssid,countof(gSsid)-1);

		WLAN_CONNECTION_PARAMETERS cp;
		cp.wlanConnectionMode = wlan_connection_mode_profile;
		cp.strProfile = ssid;
		cp.pDot11Ssid = NULL;  // all SSIDs in profile
		cp.pDesiredBssidList = NULL;
		cp.dot11BssType = dot11_BSS_type_any;
		cp.dwFlags = 0;
		if (ProgressProc(PluginNumber,TEXT("Calling WlanConnect..."),TEXT("temp"),0))
			err=-1;
		else
			err = WlanConnect(clienthandle,&FirstInterfaceGUID,&cp,NULL);
		if (err!=0) {
			WlanDeleteProfile(clienthandle,&FirstInterfaceGUID,gSsid,NULL);
			WlanCloseHandle(clienthandle,NULL);
			return err;
		}
		gClientHandle=clienthandle;
		err=-1;
		for (int i=0; i<10; i++) {
			if (ProgressProc(PluginNumber,TEXT("Connecting..."),TEXT("temp"),i*3+tries*30))
				break;
			if (GetConnectionStatus()==STATUS_CONNECTED) {
				err=0;
				break;
			}
			Sleep(500);
		}
		if (err==0)
			break;
		if (tries==0) {  //try again with WPA instead of WPA2!
			WlanDeleteProfile(clienthandle,&FirstInterfaceGUID,gSsid,NULL);
		}
	}
	if (err==0) {  // now wait until we get an IP address!
		err=-1;
		for (int i=0; i<20; i++) {
			if (ProgressProc(PluginNumber,TEXT("Getting IP address..."),TEXT("temp"),i*2+60))
				break;
			if (GetAdapterIpAddressCount(&FirstInterfaceGUID)>0) {
				err=0;
				break;
			}
			Sleep(500);
		}
	}
	if (err!=0) {
		MessageBoxA(GetActiveWindow(),"WlanConnect failed to connect!","error",MB_OK);
		DisconnectPreviouslyConnectedWlan();
		return err;
	}
	return err;
}

int GetAdapterIpAddressCount(GUID *pguid)
{
    DWORD dwSize = 0;
    DWORD dwRetVal = 0;
    unsigned int i = 0;
	char guidstr[80];

	guid_to_str(pguid, guidstr);
	strlwr(guidstr);

    // Set the flags to pass to GetAdaptersAddresses
    ULONG flags = GAA_FLAG_INCLUDE_PREFIX |
			      GAA_FLAG_SKIP_ANYCAST |
				  GAA_FLAG_SKIP_MULTICAST |
				  GAA_FLAG_SKIP_DNS_SERVER;

    ULONG family = AF_INET;
    LPVOID lpMsgBuf = NULL;

    PIP_ADAPTER_ADDRESSES pAddresses = NULL;
    ULONG outBufLen = 0;

    PIP_ADAPTER_ADDRESSES pCurrAddresses = NULL;
    PIP_ADAPTER_UNICAST_ADDRESS pUnicast = NULL;
    IP_ADAPTER_PREFIX *pPrefix = NULL;

    outBufLen = sizeof (IP_ADAPTER_ADDRESSES);
    pAddresses = (IP_ADAPTER_ADDRESSES *) malloc(outBufLen);

    // Make an initial call to GetAdaptersAddresses to get the 
    // size needed into the outBufLen variable
    if (GetAdaptersAddresses(family, flags, NULL, pAddresses, &outBufLen)
        == ERROR_BUFFER_OVERFLOW) {
        free(pAddresses);
        pAddresses = (IP_ADAPTER_ADDRESSES *) malloc(outBufLen);
    }

    if (pAddresses == NULL)
        return -1;
    
    dwRetVal = GetAdaptersAddresses(family, flags, NULL, pAddresses, &outBufLen);

    if (dwRetVal == NO_ERROR) {
        // In the returned list of all IP adapters check against the field AdapterName, which contains the interface GUID.
		// Then check the list FirstUnicastAddress for all ("normal") IP addresses on the interface. 
        pCurrAddresses = pAddresses;
        while (pCurrAddresses) {
			strlwr(pCurrAddresses->AdapterName);
			if (strstr(pCurrAddresses->AdapterName,guidstr)) {
				pUnicast = pCurrAddresses->FirstUnicastAddress;
				if (pUnicast != NULL) {
					i=0;
					while (pUnicast != NULL) {
						SOCKET_ADDRESS addr=pUnicast->Address;
						if (addr.iSockaddrLength==16)  // is it 169.254 -> no DHCP data! -> wait
							if (addr.lpSockaddr->sa_data[2]!=-87 && 
								addr.lpSockaddr->sa_data[3]!=-2)
								i++;
						pUnicast = pUnicast->Next;
					}
					if (i>0) {
						free(pAddresses);
						return i;
					}
				}
			}
			pCurrAddresses = pCurrAddresses->Next;
		}
	}
	free(pAddresses);
	return 0;
}

void DisconnectPreviouslyConnectedWlan()
{
	if (gClientHandle!=0) {
		WlanDisconnect(gClientHandle,&FirstInterfaceGUID,NULL);
		WlanDeleteProfile(gClientHandle,&FirstInterfaceGUID,gSsid,NULL);
		WlanCloseHandle(gClientHandle,NULL);
		gClientHandle=0;
	}
}


