library ex2fs;

uses
  windows,
  sysutils,
  plugdebug,
  ex2 in 'ex2.pas',
  classes,
  fsplugin,
  reiser in 'reiser.pas',
  LinuxFS in 'LinuxFS.pas',
  WinIOCTL in 'WinIOCTL.pas',
  Partition in 'Partition.pas',
  RegUtil in 'RegUtil.pas',
  BinFile in 'BinFile.pas',
  time in 'time.pas',
  INode in 'INode.pas',
  Blocks in 'Blocks.pas',
  Directory in 'Directory.pas',
  Utils in 'Utils.pas',
  LinuxFile in 'LinuxFile.pas',
  BlockDev in 'BlockDev.pas',
  DiskIO in '95Thunk\DiskIO.pas',
  QTThunkU in '95Thunk\QTThunkU.pas',
  users in 'users.pas',
  Native in 'Native.pas',
  utf8 in 'utf8.pas',
  plugpropdlg in 'plugpropdlg.pas',
  cunicode in 'cunicode.pas';

{$R tux.res}

const
   Explore2fsVersion = '1.00 pre 4';
   DebugOff=0;
   DebugLow=1;
   DebugHigh=2;
   DefPluginTitle='Linux-drives';
   wdirtypemax=1024;

var
   lockfile       : THandle=invalid_handle_value;
   LockFileName   : String;


type
// Each node in the TreeView is represented by one of these TNodeData objects
  TNodeData = class
  public
    INode         : ULONG;
    ParentINode   : ULONG;
    FileName      : String;
    Loaded        : Boolean;
    Partition     : TLinuxPartition;

    mode          : ULONG;
    uid           : USHORT;	// Owner Uid */
    gid           : USHORT;	// Owner Uid */
    size          : _LARGE_INTEGER;		// Size in bytes */
    mtime         : ULONG;		// Modification time */
    flags         : ULONG;		// File flags */
    Children      : TList;    // This keeps a list of (TNodeData objects for) all the children in the directory
    ModeString    : String;
//    TreeNode : TTreeNode;
    constructor Create;
    destructor Destroy; override;
  end;

constructor TNodeData.Create;
begin
   Loaded := False;
   Children := nil;
end;

destructor TNodeData.Destroy;
var
   i : Integer;
begin
   if Assigned(Children) then
   begin
      for i := 0 to Children.Count - 1 do
      begin
         TNodeData(Children[i]).Free;
      end;
      Children.Free;
   end;
end;

function countdots(buf:pchar):integer;
var p:pchar;
begin
  p:=buf;
  result:=0;
  repeat
    p:=StrScan(p,'.');
    if p<>nil then begin
      inc(p);
      inc(result);
    end;
  until p=nil;
end;

function filematch(swild,slbox:pchar):boolean;
var pattern,buffer:array[0..259] of char;
    ppat,pbuf,pendbuf,PosOfStar:pchar;
    failed:boolean;
begin
  strlcopy(pattern,swild,sizeof(pattern)-1);
  ansiupper(pattern);
  strlcopy(buffer,slbox,sizeof(pattern)-1);
  ansiupper(buffer);
  result:=false;
  failed:=false;
  ppat:=pattern;
  pbuf:=buffer;
  pendbuf:=strend(pbuf);
  PosOfStar:=pbuf;
  repeat
    if (ppat[0]=#0) and (pbuf[0]=#0) then result:=true
    else if ppat[0]='*' then begin {just skip to next}
      PosOfStar:=pbuf;
      inc(ppat);
      if ppat[0]=#0 then result:=true;    {* am Schluss bedeutet full Match!}
    end else begin
      if ((ppat[0]='?') and (pbuf[0]<>#0)) or (ppat[0]=pbuf[0]) then begin {Match!}
        inc(ppat);
        inc(pbuf);
      end else if (pbuf[0]=#0) and (ppat[0]='.') and (ppat[1]='*') and (ppat[2]=#0) then
        result:=true         {xyz.* matches also xyz}
      else if (pbuf[0]=#0) and (ppat[0]='.') and (ppat[1]=#0) then begin
        {Spezialfall: '.' am Ende bedeutet, dass buffer und pattern gleich viele '.' enthalten sollen!}
        ppat[0]:=#0;  {last pattern-dot doesn't count!}
        result:=countdots(buffer)=countdots(pattern);
        failed:=not result;
      end else begin  {Backtrack!}
        while (ppat>pattern) and (ppat[0]<>'*') do
          dec(ppat);
        if ppat[0]='*' then begin
          inc(ppat);
          inc(PosOfStar);
          if PosOfStar>pendbuf then failed:=true;
          pbuf:=PosOfStar;
        end else
          failed:=true;
      end;
    end;
  until result or failed;
end;

function MultiFileMatch(wild,name:pchar):boolean;
var p0:pchar;

 function strtok2(name:pchar):pchar;
 var p1,p2,p3:pchar;
 begin
   if name<>nil then p0:=name;
   if p0=nil then begin
     strtok2:=p0;
     exit
   end;
   p1:=StrScan(p0,'"');
   p2:=StrScan(p0,' ');
   p3:=StrScan(p0,';');
   if p3<>nil then if p2=nil then p2:=p3
     else if longint(p2)>longint(p3) then p2:=p3;
   if (p1=nil) or (p2<>nil) and (longint(p1)>longint(p2)) then begin
     strtok2:=p0;
     p0:=p2;
     if p0<>nil then begin
       p0[0]:=#0;
       inc(p0);
     end;
   end else begin      {Anfhrungszeichen!}
     p3:=StrScan(@p1[1],'"');
     if p3=nil then p3:=strend(p1)
       else begin
         p3[0]:=#0;
         inc(p3);
         while p3[0]=' ' do inc(p3);
       end;
     strtok2:=@p1[1];
     p0:=p3;
   end;
 end;

var sincl:array[0..2*MAX_PATH-1] of char;
    swild,p:pchar;
    io:boolean;

begin
  io:=false;
  strlcopy(sincl,wild,sizeof(sincl)-1);
  {first, check for | symbol, all behind it is negated!}
  p:=strscan(sincl,'|');
  if p<>nil then begin
    p[0]:=#0;
    if p=sincl then io:=true;
    inc(p);
    while p[0]=' ' do inc(p);
  end;
  swild:=strtok2(sincl);
  {included files}
  while (swild<>nil) and not io do begin
    if filematch(swild,name) then io:=true;
    swild:=strtok2(nil);
  end;
  {excluded files}
  if io and (p<>nil) then begin
    swild:=strtok2(p);
    while (swild<>nil) and io do begin
      if filematch(swild,name) then io:=false;
      swild:=strtok2(nil);
    end;
  end;
  MultiFileMatch:=io;
end;

var { Private declarations }
    Partitions    : TList;
    RootNodes     : TList;

    TempDir       : String;

function GetTempDir : String;
   function AdjustString(S : String) : String;
   var
      i : Integer;
   begin
      for i := 1 to Length(S) do
      begin
         if S[i] = #0 then
         begin
            Result := Copy(S, 1, i - 1);
            break;
         end;
      end;
   end;
begin
   if Length(TempDir) = 0 then
   begin
      SetLength(TempDir, 256);
      GetTempPath(256, PChar(TempDir));
      TempDir := AdjustString(TempDir);
   end;
   Result := TempDir;
end;

procedure PruneNode(Data : TNodeData);
var
   i     : Integer;
begin
   if Assigned(Data) then
   begin
      if Assigned(Data) then
      begin
         if Assigned(Data.Children) then
         begin
            for i := 0 to Data.Children.Count - 1 do
            begin
               PruneNode(Data.Children[i]);
               TNodeData(Data.Children[i]).Free;
            end;
            Data.Children.Free;
            Data.Children := nil;
         end;
      end;
   end;
end;

procedure AddPartitionToNodelist(Partition : TLinuxPartition);
var
   Data  : TNodeData;
begin
   Data                 := TNodeData.Create;
   Data.INode           := EXT2_ROOT_INO;
   Data.ParentINode     := EXT2_ROOT_INO; // just in case
   Data.Partition       := Partition;
   Data.FileName        := Partition.Description;
   Data.mode            := S_IFDIR;
   RootNodes.Add(Data);
end;

function ScanForLinux(Partitions : TList): integer;
var
   List     : TPartitionList;
   Linux    : TLinuxPartition;
   i        : Integer;
begin
   Debug('Looking for Linux Partitions', DebugLow);
   List := TPartitionList.Create;
   List.ScanFloppy := false;
   try
      // look for real PARTITION_LINUX partitions
      if OSis95 then
      begin
         Debug('Using Win95_Scan with EI13 = ' + IntToStr(Integer(true{Options.UseEI13})), DebugHigh);
         List.Win95_Scan(PARTITION_LINUX, true{Options.UseEI13});
      end
      else
      begin
         if false{Options.UseNative} then
         begin
            Debug('Using NT_Scan (NT Native IO)', DebugHigh);
            List.NT_Scan;
         end
         else
         begin
            Debug('Using LL_Scan (Low Level for NT)', DebugHigh);
            List.LL_Scan(PARTITION_LINUX);
         end;
      end;

      // do we have a non-standard partition type to look for?
(*      if (Options.NonStandard > 0) and (Options.NonStandard <> PARTITION_LINUX) and (Options.UseNative = False) then
      begin
         if List.Count > 0 then // this is bad logic, but the NonStandard is a hack too
         begin
            List.ScanFloppy := False;
         end;
         Debug('Scanning for non-standard partition type 0x' + IntToHex(Options.NonStandard, 2), DebugOff);
         if OSis95 then
         begin
            List.Win95_Scan(Options.NonStandard, Options.UseEI13);
         end
         else
         begin
            List.LL_Scan(Options.NonStandard);
         end;
      end; *)

      Debug('Found ' + IntToStr(List.Count) + ' Linux partitions', DebugLow);
      Result := List.Count;
      // Create a TLinuxPartition object for each linux partition found
      for i := 0 to List.Count - 1 do
      begin
         Linux := TLinuxPartition.Create(
            List.GetPartition(i).FileName,
            List.GetPartition(i).DriveNo,
            List.GetPartition(i).PartNo,
            List.GetPartition(i).StartingOffset,
            List.GetPartition(i).PartitionLength,
            List.GetPartition(i).BytesPerSector,
            'hd'{Options.Prefix},
            true{Options.UseEI13});
         Partitions.add(Linux);

      end;

   finally
      List.Free;
   end;
end;

procedure RescanPartitions;
var
   i     : Integer;
begin
   try
      if RootNodes<>nil then begin
        for i := 0 to RootNodes.Count - 1 do
        begin
           PruneNode(RootNodes[i]);
           TNodeData(RootNodes[i]).Free;
        end;
        RootNodes.Free;
        RootNodes := nil;
      end;

      for i := 0 to Partitions.Count - 1 do
      begin
         TLinuxPartition(Partitions[i]).Free;
      end;
      Partitions.Clear;
      RootNodes:=TList.Create;

      ScanForLinux(Partitions);

      for i := 0 to Partitions.Count - 1 do
      begin
         AddPartitionToNodelist(TLinuxPartition(Partitions[i]));
      end;
   finally
   end;
end;

procedure initall;
var
   Version        : TOSVersionInfo;
   VersionString  : String;
   buf,buf1       : array[0..259] of char;
begin
   // create lock file...
   LockFileName := GetTempDir + 'explore2fs.lock';
   lockfile := CreateFile(PChar(LockFileName), GENERIC_READ, 0, nil, OPEN_ALWAYS, 0, 0);
   if lockfile = INVALID_HANDLE_VALUE then
   begin
      Messagebox(0,'Explore2fs is already running.  Running more that one'#10 +
                 'instance of this program is not possible', 'error', mb_iconstop);
      //Application.Terminate;
      exit;
   end;

   try
      Partitions  := TList.Create;
      Debug('Explore2fs version ' + Explore2fsVersion, DebugOff);
      Debug('Written by John Newbigin', DebugOff);

      // Prevent error messages being displayed by NT
      SetErrorMode(SEM_FAILCRITICALERRORS);

      RootNodes  := nil;

      // find out OS version
      Version.dwOSVersionInfoSize := Sizeof(Version);
      if GetVersionEx(Version) then
      begin
         case Version.dwPlatformId of
            VER_PLATFORM_WIN32s        : VersionString := 'WIN32s';
            VER_PLATFORM_WIN32_WINDOWS : VersionString := 'Windows 95';
            VER_PLATFORM_WIN32_NT      : VersionString := 'Windows NT';
         else
            VersionString := 'Unknown OS';
         end;
         VersionString := VersionString + ' ' + IntToStr(Version.dwMajorVersion) +
                                          '.' + IntToStr(Version.dwMinorVersion) +
                                          ' build number ' + IntToStr(Version.dwBuildNumber);
         Debug(VersionString, DebugOff);
         if Version.dwPlatformId = VER_PLATFORM_WIN32_WINDOWS then
         begin
            OSis95 := True;
         end
         else
         begin
            OSis95 := False;
         end;
         OSType := Version.dwPlatformId;
      end
      else
      begin
         Debug('Could not get Version info!', DebugOff);
      end;

      RescanPartitions;
   finally
   end;
   strcopy(buf,'CONNECT \');
   LogProc(PluginNumber,MSGTYPE_CONNECT,buf);

   strcopy(buf,'Ext2 plugin: ');
   str(partitions.count,buf1);
   strcat(buf,buf1);
   strcat(buf,' partition(s) found');
   LogProc(PluginNumber,msgtype_connectcomplete,buf);
   if not OSis95 and (partitions.count=0) then
     LogProc(PluginNumber,msgtype_connectcomplete,'Make sure you run as an Administrator!');
end;

procedure doneall;
var
   i : Integer;
begin
  try
    if RootNodes<>nil then begin
      for i := 0 to RootNodes.Count - 1 do
      begin
         PruneNode(RootNodes[i]);
         TNodeData(RootNodes[i]).Free;
      end;
      RootNodes.Free;
      RootNodes := nil;
    end;
    if partitions<>nil then begin
      for i := 0 to Partitions.Count - 1 do
      begin
         TLinuxPartition(Partitions[i]).Free;
      end;
      Partitions.Clear;
    end;  
  except
    on E : Exception do
    begin
       Messagebox(0,pchar(E.Message),'ex2fs',0);
    end;
  end;
end;

procedure LoadNode(NodeData : TNodeData);
var
   Partition   : TLinuxPartition;
   INode       : TINode;
   Dir         : TDirectory;
   Data        : TNodeData;
   NewINode    : TINode;
   List        : TStringList;
   i           : Integer;
//   Links       : Integer;
begin
   if Assigned(NodeData) then
   begin
      if not NodeData.Loaded then
      begin
         try
            // read the inode
            Partition := NodeData.Partition;

            try
               Partition.Open; // Make sure - this will raise an exception if it fails

               INode := TINode.Create(Partition, NodeData.INode);
               try
                  Partition.DumpINode(INode.Info);
                  // now extract the directory info (assume it is is a dir if we are expanding it!)
                  Dir := TDirectory.Create(Partition, INode);

                  try
                     List := TStringList.Create;
                     Dir.EnumDir(List);
                     for i := 0 to List.Count - 1 do
                     begin
//                        Links := 0;
                        if not((CompareStr(List[i], '.') = 0) or (CompareStr(List[i], '..') = 0)) then
                        begin
                           Data := TNodeData.Create;
                           Data.INode              := LongInt(List.Objects[i]);
                           Data.FileName           := List[i];
                           Data.ParentINode        := NodeData.INode;
                           Data.Partition          := Partition;

                           // we have to lock the inode and read it's data....?
                           if Data.INode > 0 then
                           begin
                              NewINode := TINode.Create(Partition, Data.INode);
                              try
                                 Data.mode   := NewINode.Info.i_mode;
                                 Data.uid    := NewINode.Info.i_uid;
                                 Data.gid    := NewINode.Info.i_gid;
                                 Data.size   := NewINode.GetSize;
                                 Data.mtime  := NewINode.Info.i_mtime;
                                 Data.flags  := NewINode.Info.i_flags;
                                 Data.ModeString := NewINode.ModeString;

//                                 Links := NewINode.Info.i_links_count;
                              finally
                                 NewINode.Free;
                              end;
                           end;

                           if not Assigned(NodeData.Children) then
                           begin
                             NodeData.Children := TList.Create;
                           end;
                           NodeData.Children.Add(Data);
                        end;
                     end;
                  finally
                     Dir.Free;
                  end;

               finally
                  INode.Free;
               end;

               NodeData.Loaded := True;
            except
               on E : Exception do
               begin
                  debug('Error: ' + E.Message, 0);
                  NodeData.Loaded := False;
               end;
            end;

         finally
         end;
      end;
   end;
end;

function RecursiveFindNode(childlist:tlist;pnames:pchar):TNodeData;
var pend:pchar;
    i:integer;
    data:TNodeData;
begin
  while pnames[0]='\' do inc(pnames);  {skip consecutive backslashes}
  pend:=strscan(pnames,'\');
  if pend<>nil then begin
    pend[0]:=#0;
    inc(pend);
    if pend[0]=#0 then pend:=nil;  {Trailing backslash}
  end;
  if pnames[0]<>#0 then begin
    for i:=0 to childlist.count-1 do begin
      data:=TNodeData(childlist[i]);
      if strpas(pnames)=data.filename then begin  {Found!!!}
        if (pend=nil) then
          result:=data
        else begin
          LoadNode(data);
          if data.Partition.ReiserFs then begin
            result:=data;
            exit
          end;
          result:=RecursiveFindNode(data.children,pend);
        end;
        exit;
      end;
    end;
  end;
  result:=nil;    {Find failed!}
end;

function retrievepath(thepath:pchar):tnodedata;
var data:TNodeData;
    pathname:array[0..259] of char;
begin
  result:=nil;
  if thepath[0]<>'\' then exit;
  strlcopy(pathname,thepath+1,sizeof(pathname)-1);
  data:=RecursiveFindNode(rootnodes,pathname);
  if data<>nil then begin
    if (Data.mode and S_IFMT) = S_IFDIR then {don't load if it's a file!}
      LoadNode(data);
    result:=data;
  end;
end;

function IsReiserFs(thepath:pchar):tnodedata;
var data:TNodeData;
    pathname:array[0..259] of char;
    p:pchar;
    i:integer;
begin
  result:=nil;
  if thepath[0]<>'\' then exit;
  strlcopy(pathname,thepath+1,sizeof(pathname)-1);
  p:=strscan(pathname,'\');
  if p<>nil then p[0]:=#0;
  for i:=0 to rootnodes.count-1 do begin
    data:=TNodeData(rootnodes[i]);
    if strpas(pathname)=data.filename then begin  {Found!!!}
      if data.partition.ReiserFs then
        result:=data;
      exit;
    end;
  end;
end;

{****************************************************************************}

function FsInit(PluginNr:integer;pProgressProc:tProgressProc;pLogProc:tLogProc;
                pRequestProc:tRequestProc):integer; stdcall;
begin
  ProgressProc:=PProgressProc;
  LogProc:=PLogProc;
  RequestProc:=PRequestProc;
  PluginNumber:=PluginNr;
  result:=0;
end;

function FsInitW(PluginNr:integer;pProgressProcW:tProgressProcW;pLogProcW:tLogProcW;
                pRequestProcW:tRequestProcW):integer; stdcall;
begin
  ProgressProcW:=PProgressProcW;
  LogProcW:=PLogProcW;
  RequestProcW:=PRequestProcW;
  PluginNumber:=PluginNr;
  result:=0;
end;

type plf=^tlf;
     tlf=record
       path:array[0..259] of char;
       lastsearchitem:integer;
       nodedata:TNodeData;
       ReiserFs:boolean;
     end;

function UnixTimeToWindowsTime(mtime:longint):tFileTime;
var ft:tFileTime;
begin
  ft.dwLowDateTime:=$D53E8000;
  ft.dwHighDateTime:=$019DB1DE;
  comp(ft):=comp(ft)+10000000.0*(mtime);  {Keep universal time!} 
  result:=ft;
end;

procedure fillitemstruct(data:tnodedata;var FindData:tWIN32FINDDATA);
begin
  if data<>nil then begin
    strplcopy(FindData.cFileName,data.FileName,MAX_PATH-1);
    if (Data.mode and S_IFMT) = S_IFDIR then begin
      FindData.dwFileAttributes:=FILE_ATTRIBUTE_DIRECTORY or $80000000;
      FindData.nFileSizeLow:=0;
    end else begin
      FindData.dwFileAttributes:=$80000000;  {Signal Unix mode}
      FindData.nFileSizeLow:=Data.size.LowPart;
      FindData.nFileSizeHigh:=Data.size.HighPart;
	end;
	FindData.ftLastWriteTime:=UnixTimeToWindowsTime(Data.mtime);
	FindData.dwReserved0:=Data.mode;
  end;
end;

type li=record
          lo,hi:longint;
        end;
        TCharArray = array[char] of char;

function IniReadString(const FileName,Section, KeyName, Default: string): string;
var
  Buffer: array[0..2047] of Char;
begin
  SetString(Result, Buffer, GetPrivateProfileString(PChar(Section),
	PChar(KeyName), PChar(Default), Buffer, SizeOf(Buffer), PChar(FileName)));
end;

var CharsetsLoaded:boolean=false;
    LoadedCharset:array[boolean] of TCharArray;
    preloadisutf8:boolean=false;

function PreloadCharsetAndUtf8(Reverse: boolean): TCharArray;
var
  ChangeTableName,TestUTF8,
  s: string;
  buf:array[0..259] of char;
  i : Integer;
  C1,
  C2: char;
  SL: TStringList;
begin
  for c1 := #0 to #255 do
    Result[c1] := c1;
  SL := TStringList.Create;
  GetModuleFileName(hinstance,buf,sizeof(buf)-1);
  s := strpas(buf);
  s := ChangeFileExt(s,'.ini');
  TestUTF8:=UpperCase(IniReadString(s,'Options','UTF8','false'));
  preloadisutf8:=(TestUTF8 = 'TRUE')or(TestUTF8 = '1')or(TestUTF8 = 'YES')or(TestUTF8 = 'Y')or
     (TestUTF8 = 'T')or(TestUTF8 = 'ON');
  ChangeTableName := UpperCase(IniReadString(s,'Options','EnableXLAT','false'));
  if (ChangeTableName = 'TRUE')or(ChangeTableName = '1')or(ChangeTableName = 'YES')or(ChangeTableName = 'Y')or
     (ChangeTableName = 'T')or(ChangeTableName = 'ON') then begin
    ChangeTableName := Trim(UpperCase(IniReadString(s,'Options','XLATfile','')));
    if (ChangeTableName <> '') then begin
      s := ExtractFilePath(s)+'\'+ChangeTableName;
      SL.LoadFromFile(s);
      i := 0;
      while (i < SL.Count) do begin
        if(Uppercase(copy(SL[i],1,2))='0X') then begin // If first two characters is hex value
          C1 := char(StrToIntDef(copy(SL[i],1,4),0));
          C2 := char(StrToIntDef(copy(SL[i],6,4),ord(C1)));
          if not Reverse then
            Result[C1] := C2
          else
            Result[C2] := C1;
        end;
        inc(i);
      end;
    end;
  end;
end;

function GetCharset(Reverse: boolean): TCharArray;
begin
  if not CharsetsLoaded then begin  {Caching!}
    LoadedCharset[true]:=PreloadCharsetAndUtf8(true);
    LoadedCharset[false]:=PreloadCharsetAndUtf8(false);
    CharsetsLoaded:=true;
  end;
  result:=LoadedCharset[Reverse];
end;

function ChangeFileNameToInternal(const aFn: string): string;
var
  i		: Integer;
  Charset	: TCharArray;
begin
  Result := aFn;
  i := 0;
  Charset:=GetCharset(true);
  if preloadisutf8 then begin
    Result:=AnsiStringToUtf8(aFn)
  end else
  while (i < Length(Result)) do
  begin
    Result[i] := Charset[Result[i]];
    inc(i);
  end;
end;

function wstrlen(wp:pwidechar):integer;
begin
  if wp=nil then result:=0
  else begin
    result:=0;
    while wp[result]<>#0 do
      inc(result);
  end;
end;

procedure ChangeFileNameToInternalW(PathA:pchar;PathW:pwidechar;maxlen:integer);
var
  i		: Integer;
  Charset	: TCharArray;
begin
  i := 0;
  Charset:=GetCharset(true);
  if preloadisutf8 then begin
    ConvertUTF16toUTF8(PathW,PathW+wstrlen(PathW)+1,
      PathA,PathA+maxlen-1);
  end else begin
    WideCharToMultiByte(CP_ACP,0,PathW,-1,PathA,maxlen,nil,nil);
    while (i < strlen(PathA)) do
    begin
      PathA[i] := Charset[PathA[i]];
      inc(i);
    end;
  end;
end;

procedure ChangeFileItemCodePage(var FindData:tWIN32FINDDATA);
var
  i		: Integer;
  Charset	: TCharArray;
begin
  i := 0;
  Charset := GetCharset(false);
  if preloadisutf8 then begin
    strplcopy(FindData.cFileName,Utf8StringToAnsi(FindData.cFileName),sizeof(FindData.cFileName)-1)
  end else
  while (i < MAX_PATH)and(FindData.cFileName[i] <> #0) do begin
    FindData.cFileName[i] := Charset[FindData.cFileName[i]];
    inc(i);
  end;
end;

procedure ChangeFileItemCodePageW(nameW:pwidechar;nameA:pchar;maxlen:integer);
var
  i		: Integer;
  Charset	: TCharArray;
  unusedptr: pwidechar;
  tempbuf:array[0..wdirtypemax] of char;
begin
  i := 0;
  Charset := GetCharset(false);
  if preloadisutf8 then begin
    ConvertUTF8toUTF16(NameA,NameA+strlen(NameA)+1,NameW,NameW+maxlen-1,nil,unusedptr);
  end else begin
    strlcopy(tempbuf,NameA,wdirtypemax);
    while (i < wdirtypemax) and (tempbuf[i] <> #0) do begin
      tempbuf[i] := Charset[tempbuf[i]];
      inc(i);
    end;
    MultiByteToWideChar(CP_ACP,0,tempbuf,-1,NameW,maxlen);
  end;
end;

function FsFindFirstInt(path :pchar;var FindData:tWIN32FINDDATA):thandle; stdcall;
var data:TNodeData;
	buf:array[0..MAX_PATH-1] of char;
	lf:plf;
	pfi:pfileiteminfostruct;
	ofs:comp;
	p:pchar;
begin
  CharsetsLoaded:=false;      {Reload for each subdir!}
  try
    strcopy(buf,'Ext2 plugin: DIR ');
    strcat(buf,path);
    LogProc(PluginNumber,msgtype_details,buf);
    fillchar(FindData,sizeof(tWIN32FINDDATA),#0);
    if strcomp(Path,'\')=0 then begin
      FindData.dwFileAttributes:=FILE_ATTRIBUTE_DIRECTORY;
      FindData.ftLastWriteTime.dwHighDateTime:=$FFFFFFFF;
      FindData.ftLastWriteTime.dwLowDateTime:=$FFFFFFFE;
      new(lf);
      strcopy(lf.Path,Path);
      lf.nodedata:=nil;
      lf.lastsearchitem:=0;
      lf.ReiserFs:=false;
      if Partitions=nil then begin
        initall;
      end;
      if RootNodes.Count>0 then begin
        data:=TNodeData(RootNodes[0]);
        strplcopy(FindData.cFileName,data.FileName,MAX_PATH-1);
        result:=thandle(lf);
      end else begin
        dispose(lf);
        result:=INVALID_HANDLE_VALUE;
        SetLastError(ERROR_FILE_NOT_FOUND);
      end;
    end else begin
      if Partitions=nil then begin
        initall;
      end;
      data:=IsReiserFs(Path);
      if data=nil then begin
        try
          data:=retrievepath(Path);
        except
          data:=nil;
        end;
      end;
      if (data<>nil) and data.partition.ReiserFs then begin
        p:=strscan(path+1,'\');
        if p<>nil then
          strlcopy(buf,p,sizeof(buf)-1)
        else
          strcopy(buf,'/');
        p:=buf;
        while p[0]<>#0 do begin
          if p[0]='\' then p[0]:='/';
          inc(p);
        end;
        if OSType = VER_PLATFORM_WIN32_NT then
          ofs:=data.partition.Start.Quadpart
        else
          ofs:=data.partition.Start.Quadpart*data.partition.SectorSize;
        if @ReiserReadDirList<>nil then
          pfi:=ReiserReadDirList((data.partition.Drive and 127),li(ofs).lo,li(ofs).hi,buf)
        else begin
          Debug('Function ReiserReadDirList not available, cannot access Reiser FS!', DebugOff);
          pfi:=nil;
        end;
        if (pfi<>nil) and (pfi.size_lo>0) then begin {First record contains number of files in size_lo field}
          new(lf);
          lf.ReiserFs:=true;
          lf.nodedata:=pointer(pfi);
          lf.lastsearchitem:=1;
          inc(pfi);
          strplcopy(FindData.cFileName,pfi.filename,MAX_PATH-1);
          if (pfi.mode and S_IFMT) = S_IFDIR then begin
            FindData.dwFileAttributes:=FILE_ATTRIBUTE_DIRECTORY or $80000000;
            FindData.nFileSizeLow:=0;
            FindData.nFileSizeHigh:=0;
          end else begin
            FindData.dwFileAttributes:=$80000000;  {Signal Unix mode}
            FindData.nFileSizeLow:=pfi.size_lo;
            FindData.nFileSizeHigh:=pfi.size_hi;
          end;
          FindData.ftLastWriteTime:=UnixTimeToWindowsTime(pfi.mtime);
          FindData.dwReserved0:=pfi.mode;
          result:=thandle(lf);
        end else begin
          result:=INVALID_HANDLE_VALUE;
          if (pfi<>nil) then
            SetLastError(ERROR_NO_MORE_FILES)    {Empty dir!}
          else
            SetLastError(ERROR_FILE_NOT_FOUND);
        end;
      end else begin
        if (data=nil) or (data.children=nil) or (data.children.count=0) then begin
          if (data<>nil) and ((Data.mode and S_IFMT) = S_IFDIR) then
            SetLastError(ERROR_NO_MORE_FILES)    {Empty dir!}
          else
            SetLastError(ERROR_FILE_NOT_FOUND);
          result:=INVALID_HANDLE_VALUE
        end else begin
          new(lf);
          strcopy(lf.Path,Path);
          lf.nodedata:=data;
          lf.lastsearchitem:=0;
          lf.ReiserFs:=false;
          fillitemstruct(data.children[0],FindData);
          result:=thandle(lf);
        end;
      end;
    end;
  except        {Don't let exceptions out of the DLL!}
    SetLastError(ERROR_FILE_NOT_FOUND);
    result:=INVALID_HANDLE_VALUE;
  end;
end;

function FsFindFirst(path :pchar;var FindData:tWIN32FINDDATA):thandle; stdcall;
var path2:array[0..MAX_PATH-1] of char;
begin
  Path2[0] := #0;
  StrPLCopy(Path2,ChangeFileNameToInternal(String(Path)),MAX_PATH-1);
  Path := Path2;
  result:=FsFindFirstInt(path,FindData);
  if result<>INVALID_HANDLE_VALUE then
    ChangeFileItemCodePage(FindData);
end;

function FsFindFirstW(path :pwidechar;var FindData:tWIN32FINDDATAW):thandle; stdcall;
var pathA:array[0..wdirtypemax] of char;
    FindDataA:tWIN32FINDDATA;
begin
  PathA[0] := #0;
  ChangeFileNameToInternalW(PathA,Path,wdirtypemax);
  result:=FsFindFirstInt(pathA,FindDataA);
  if result<>INVALID_HANDLE_VALUE then begin
    move(FindDataA,FindData,sizeof(FindDataA));  {Copy all non-text data}
    ChangeFileItemCodePageW(FindData.cFileName,FindDataA.cFileName,MAX_PATH);
  end;
end;

function FsFindNextInt(Hdl:thandle;var FindData:tWIN32FINDDATA):bool; stdcall;
var data:TNodeData;
    pfi:pfileiteminfostruct;
    lf:plf;
begin
  try
    result:=false;
    fillchar(FindData,sizeof(tWIN32FINDDATA),#0);
    if (Hdl=1) or (hdl=0) or (hdl=invalid_handle_value) then result:=false
    else begin
      lf:=plf(hdl);
      if lf.nodedata=nil then begin  {Root}
        inc(lf.lastsearchitem);
        if lf.lastsearchitem<RootNodes.Count then begin
          data:=TNodeData(RootNodes[lf.lastsearchitem]);
          FindData.dwFileAttributes:=FILE_ATTRIBUTE_DIRECTORY;
          FindData.ftLastWriteTime.dwHighDateTime:=$FFFFFFFF;
          FindData.ftLastWriteTime.dwLowDateTime:=$FFFFFFFE;
          strplcopy(FindData.cFileName,data.FileName,MAX_PATH-1);
          result:=true;
        end;
      end else begin
        inc(lf.lastsearchitem);
        if lf.ReiserFs then begin
          pfi:=pointer(lf.nodedata);
          if lf.lastsearchitem<=pfi.size_lo then begin  {First record contains number of files in size_lo field}
            inc(pfi,lf.lastsearchitem);
            strplcopy(FindData.cFileName,pfi.filename,MAX_PATH-1);
            if (pfi.mode and S_IFMT) = S_IFDIR then begin
              FindData.dwFileAttributes:=FILE_ATTRIBUTE_DIRECTORY or $80000000;
              FindData.nFileSizeHigh:=0;
              FindData.nFileSizeLow:=0;
            end else begin
              FindData.dwFileAttributes:=$80000000;  {Signal Unix mode}
              FindData.nFileSizeLow:=pfi.size_lo;
              FindData.nFileSizeHigh:=pfi.size_hi;
            end;
            FindData.ftLastWriteTime:=UnixTimeToWindowsTime(pfi.mtime);
            FindData.dwReserved0:=pfi.mode;
            result:=true;
          end;
        end else
          if lf.lastsearchitem<lf.nodedata.children.Count then begin
            fillitemstruct(lf.nodedata.children[lf.lastsearchitem],FindData);
            result:=true;
          end;
      end;
	end;
  except        {Don't let exceptions out of the DLL!}
    result:=false;
  end;
end;

function FsFindNext(Hdl:thandle;var FindData:tWIN32FINDDATA):bool; stdcall;
begin
  result:=FsFindNextInt(Hdl,FindData);
  if result then
    ChangeFileItemCodePage(FindData);
end;

function FsFindNextW(Hdl:thandle;var FindData:tWIN32FINDDATAW):bool; stdcall;
var FindDataA:tWIN32FINDDATA;
begin
  result:=FsFindNextInt(Hdl,FindDataA);
  if result then begin
    move(FindDataA,FindData,sizeof(FindDataA));  {Copy all non-text data}
    ChangeFileItemCodePageW(FindData.cFileName,FindDataA.cFileName,MAX_PATH);
  end;
end;

function FsFindClose(Hdl:thandle):integer; stdcall;
var lf:plf;
begin
  if (Hdl=1) or (hdl=0) or (hdl=invalid_handle_value) then result:=-1
  else begin
    lf:=plf(hdl);
    if (lf.nodedata<>nil) and (lf.ReiserFs) and (@ReiserFreeDirList<>nil) then begin
      ReiserFreeDirList(pointer(lf.nodedata));
    end;
    dispose(lf);
    result:=0;
  end;
end;

procedure replaceslashbybackslash(var s:string);
var i:integer;
begin
  for i:=1 to length(s) do
    if s[i]='/' then s[i]:='\';
end;

procedure formcorrectlinkpath(var Resolved:string;lastpath:string);
var i:integer;
    p:pchar;
begin
  replaceslashbybackslash(Resolved);
  if (resolved<>'') and (resolved[1]='\') then begin  {Absolute path?}
    i:=pos('\',copy(lastpath,2,4096));
    Resolved:=copy(lastpath,1,i)+resolved;
  end else begin  {Relative path}
    p:=strrscan(pchar(lastpath),'\');
    if p<>nil then begin
      lastpath:=copy(lastpath,1,p-pchar(lastpath));  {Truncate string}
      while copy(Resolved,1,3)='..\' do begin
        p:=strrscan(pchar(lastpath),'\');
        if p<>nil then
          lastpath:=copy(lastpath,1,p-pchar(lastpath));  {cut updir}
        Resolved:=copy(Resolved,4,4096);
      end;
      Resolved:=lastpath+'\'+Resolved;
    end;
  end;
end;

function FsGetFileInt(RemoteName:pchar;LocalName:pwidechar;CopyFlags:integer;
                   ri:pRemoteInfo):integer; stdcall;
var
   INode : TINode;
   data : TNodeData;
   err,md,numredirects:integer;
   Resolved:string;
   lastpath:string;
   match:array[0..2*MAX_PATH-1] of char;
   buf:array[0..MAX_PATH-1] of char;
   LocalNameA: array[0..MAX_PATH-1] of char;
   p:pchar;
   ofs:comp;
   translate:boolean;
   datetime:tfiletime;

begin
  try
	data:=IsReiserFs(RemoteName);
    if data<>nil then begin
      if CopyFlags and FS_COPYFLAGS_OVERWRITE=0 then
        if testfiletypeT(LocalName)<>0 then begin
          result:=FS_FILE_EXISTS;
          exit
        end;  
      p:=strscan(RemoteName+1,'\');
      if p<>nil then
        strlcopy(buf,p,sizeof(buf)-1)
      else
        strcopy(buf,'/');
      p:=buf;
      while p[0]<>#0 do begin
        if p[0]='\' then p[0]:='/';
        inc(p);
      end;
      if (@ProgressProc<>nil) then
        if ProgressProc(PluginNumber,'',RemoteName,0)<>0 then begin
          result:=ERROR_OPERATION_ABORTED;
          exit;
        end;
      if OSType = VER_PLATFORM_WIN32_NT then
		ofs:=data.partition.Start.Quadpart
      else
        ofs:=data.partition.Start.Quadpart*data.partition.SectorSize;
      result:=FS_FILE_NOTFOUND;
      if Usys and (@ReiserGetFileW<>nil) then begin
        if ReiserGetFileW((data.partition.Drive and 127),
                       li(ofs).lo,li(ofs).hi,buf,LocalName) then begin
          result:=0;
          if (@ProgressProcW<>nil) then
            if ProgressProc(PluginNumber,'',RemoteName,100)<>0 then begin
              result:=ERROR_OPERATION_ABORTED;
              exit;
            end;
        end;
      end else if @ReiserGetFile<>nil then begin
        walcopy(LocalNameA,LocalName,sizeof(LocalNameA)-1);
        if ReiserGetFile((data.partition.Drive and 127),
                       li(ofs).lo,li(ofs).hi,buf,LocalNameA) then begin
          result:=0;
          if (@ProgressProc<>nil) then
            if ProgressProc(PluginNumber,'',RemoteName,100)<>0 then begin
              result:=ERROR_OPERATION_ABORTED;
              exit;
            end;
        end;
      end;
      exit;
    end;
    data:=retrievepath(RemoteName);
    md:=(Data.mode and S_IFMT);
    if (data=nil) or (data<>nil) and (md<>S_IFREG) and (md<>S_IFLNK) then begin
      result:=FS_FILE_NOTFOUND;
    end else begin
      // lock the inode
      INode := TINode.Create(Data.Partition, Data.INode);
      try
        numredirects:=0;
		    lastpath:=strpas(RemoteName);
        while (md=s_IFLNK) and (numredirects<5) do begin
          inc(numredirects);
          if INode.ResolveLink(Resolved)=0 then begin {Follow the link!}
            formcorrectlinkpath(Resolved,lastpath);
            lastpath:=resolved;
            data:=retrievepath(pchar(Resolved));
            if data=nil then begin
              result:=FS_FILE_NOTFOUND;
              exit;
            end;
            INode.Free;
            INode := TINode.Create(Data.Partition, Data.INode);
          end;
          md:=(Data.mode and S_IFMT);
        end;
        if md<>s_IFREG then begin     {Still not resolved, or not pointing to file}
          result:=FS_FILE_NOTFOUND;
          exit;
        end;
        if CopyFlags and FS_COPYFLAGS_OVERWRITE<>0 then
          DeleteFileT(LocalName);

        translate:=transfermode='A';
        if transfermode='X' then begin
          strlcopy(match,asciiextensions,sizeof(match)-1);
          translate:=multifilematch(match,remotename);
        end;
        if ri<>nil then
          datetime:=ri^.LastWriteTime
        else
          datetime.dwHighDateTime:=0;
        err:=INode.SaveToFileT(RemoteName,LocalName, ProgressProc, ProgressProcW, translate, datetime);
        case err of
          0:begin
              result:=FS_FILE_OK;
            end;
          ERROR_FILE_EXISTS:result:=FS_FILE_EXISTS;
          ERROR_OPERATION_ABORTED:result:=FS_FILE_USERABORT;
          ERROR_WRITE_FAULT:result:=FS_FILE_WRITEERROR;
        else
          result:=FS_FILE_READERROR;
        end;
      finally
        INode.Free;
      end;
    end;
  except        {Don't let exceptions out of the DLL!}
    result:=FS_FILE_NOTFOUND;
  end;
end;

function FsGetFile(RemoteName,LocalName:pchar;CopyFlags:integer;
                   ri:pRemoteInfo):integer; stdcall;
var path2:array[0..MAX_PATH-1] of char;
    LocalNameW:array[0..MAX_PATH-1] of widechar;
begin
  StrPLCopy(Path2,ChangeFileNameToInternal(String(RemoteName)),MAX_PATH-1);
  MultiByteToWideChar(CP_ACP,0,LocalName,-1,LocalNameW,MAX_PATH-1);
  result:=FsGetFileInt(Path2,LocalNameW,CopyFlags,ri);
end;

function FsGetFileW(RemoteName,LocalName:pwidechar;CopyFlags:integer;
                   ri:pRemoteInfo):integer; stdcall;
var RemoteNameA:array[0..wdirtypemax] of char;
begin
  ChangeFileNameToInternalW(RemoteNameA,RemoteName,wdirtypemax);
  result:=FsGetFileInt(RemoteNameA,LocalName,CopyFlags,ri);
end;

function FsExecuteFile(MainWin:thandle;RemoteName,Verb:pchar):integer; stdcall;
var
   INode : TINode;
   data : TNodeData;
   md,numredirects:integer;
   Resolved:string;
   lastpath:string;
   RemoteName2: array[0..MAX_PATH-1] of char;
begin
	RemoteName2[0] := #0;
	StrPLCopy(RemoteName2,ChangeFileNameToInternal(String(RemoteName)),MAX_PATH-1);
  try
    if stricomp(Verb,'properties')=0 then begin
	  data:=retrievepath(RemoteName2);
	  if (data=nil) then
        result:=FS_EXEC_ERROR
      else begin
        INode := TINode.Create(Data.Partition, Data.INode);
        try
          ShowFileProperties(MainWin,INode,RemoteName2);
        finally
          INode.Free;
        end;
        result:=FS_EXEC_OK;
      end;
    end else if strlicomp(Verb,'mode',4)=0 then begin
      transfermode:=upcase(verb[5]);
      if transfermode='X' then
        strlcopy(asciiextensions,@verb[6],sizeof(asciiextensions)-1);
       result:=FS_EXEC_OK;
    end else if stricomp(Verb,'open')=0 then begin
      data:=retrievepath(RemoteName2);
      md:=(Data.mode and S_IFMT);
      if (data=nil) then
        result:=FS_EXEC_ERROR
      else if (md=S_IFREG) then
        result:=FS_EXEC_YOURSELF
      else if (md<>S_IFLNK) then
        result:=FS_EXEC_ERROR
      else begin   {Follow the link!}
        // lock the inode
        INode := TINode.Create(Data.Partition, Data.INode);
        try
          numredirects:=0;
		  lastpath:=strpas(RemoteName2);
          while (md=s_IFLNK) and (numredirects<5) do begin
            inc(numredirects);
            if INode.ResolveLink(Resolved)=0 then begin {Follow the link!}
              formcorrectlinkpath(Resolved,lastpath);
              LogProc(PluginNumber,msgtype_details,pchar(resolved));
              lastpath:=resolved;
              data:=retrievepath(pchar(Resolved));
              if data=nil then begin
                result:=FS_EXEC_ERROR;
                exit;
              end;
              INode.Free;
              INode := TINode.Create(Data.Partition, Data.INode);
            end;
            md:=(Data.mode and S_IFMT);
          end;
          if md=s_IFREG then
            result:=FS_EXEC_YOURSELF
          else if md=s_IFDIR then begin
            strplcopy(RemoteName2,resolved,MAX_PATH-1);
            result:=FS_EXEC_SYMLINK;
          end else
            result:=FS_EXEC_ERROR;  {Still not resolved, or a device}
        finally
          INode.Free;
        end;
      end; {Follow the link!}
    end else
      result:=FS_EXEC_YOURSELF;
  except        {Don't let exceptions out of the DLL!}
    result:=FS_EXEC_ERROR;
  end;
end;

function FsDisconnect(DisconnectRoot:pchar):bool; stdcall;
begin
  try
    doneall;
  except        {Don't let exceptions out of the DLL!}
  end;
  if lockfile<>invalid_handle_value then begin
    CloseHandle(lockfile);
    lockfile:=invalid_handle_value;
    DeleteFile(lockfilename);
  end;
  Partitions:=nil;
  RootNodes:=nil;
  result:=true;
end;

procedure FsGetDefRootName(DefRootName:pchar;maxlen:integer); stdcall;
begin
  strlcopy(DefRootName,DefPluginTitle,maxlen-1);
end;


exports FsInit,FsInitW,FsFindFirst,FsFindFirstW,FsFindNext,FsFindNextW,FsFindClose,FsGetFile,FsGetFileW,FsExecuteFile,
        FsDisconnect,FsGetDefRootName;

var SaveExit:tfarproc;

procedure LibExit;
begin
  try
    doneall;
  except        {Don't let exceptions out of the DLL!}
  end;
  ExitProc := SaveExit;
end;

{$E wfx}

begin
  Partitions:=nil;
  RootNodes:=nil;
  SaveExit := ExitProc;	{ Kette der Exit-Prozeduren speichern }
  ExitProc := @LibExit;	{ Exit-Prozedur LibExit installieren }
end.


