<?php
/**
 * Describe child theme functions
 *
 * @package Wisdom Blog
 * @subpackage Wisdom Minimal
 * 
 */

/*-------------------------------------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'wisdom_minimal_setup' ) ) :
    /**
     * Sets up theme defaults and registers support for various WordPress features.
     *
     * Note that this function is hooked into the after_setup_theme hook, which
     * runs before the init hook. The init hook is too late for some features, such
     * as indicating support for post thumbnails.
     */
    function wisdom_minimal_setup() {

        // This theme uses wp_nav_menu() in one location.
        register_nav_menus( array(
            'wisdom_minimal_top_menu' => esc_html__( 'Top Menu', 'wisdom-minimal' )
        ) );

        // theme version
        $wisdom_minimal_theme_info = wp_get_theme();
        $GLOBALS['wisdom_minimal_version'] = $wisdom_minimal_theme_info->get( 'Version' );

    }
    endif;

add_action( 'after_setup_theme', 'wisdom_minimal_setup' );

/*-------------------------------------------------------------------------------------------------------------------------------*/

if ( ! function_exists( 'wisdom_minimal_fonts_url' ) ) :
    /**
     * Register Google fonts for News Vibrant Mag.
     *
     * @return string Google fonts URL for the theme.
     * @since 1.0.0
     */
    function wisdom_minimal_fonts_url() {

        $fonts_url = '';
        $font_families = array();

        /*
         * Translators: If there are characters in your language that are not supported
         * by Noto Serif, translate this to 'off'. Do not translate into your own language.
         */
        if ( 'off' !== _x( 'on', 'Noto Serif font: on or off', 'wisdom-minimal' ) ) {
            $font_families[] = 'Noto Serif:400,700';
        }

        if( $font_families ) {
            $query_args = array(
                'family' => urlencode( implode( '|', $font_families ) ),
                'subset' => urlencode( 'latin,latin-ext' ),
            );

            $fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
        }

        return $fonts_url;
    }
endif;

/*-------------------------------------------------------------------------------------------------------------------------------*/
    
if( ! function_exists( 'wisdom_minimal_customize_register' ) ) :
    /**
     * Managed the theme default color
     */
    function wisdom_minimal_customize_register( $wp_customize ) {
        
        global $wp_customize;

        $wp_customize->get_setting( 'wisdom_blog_theme_color' )->default = '#f7b4b4';

        /**
         * Add Top Header Section
         */
        $wp_customize->add_section(
            'wisdom_minimal_top_header_section',
            array(
                'priority'       => 10,
                'panel'          => 'wisdom_blog_header_settings_panel',
                'capability'     => 'edit_theme_options',
                'theme_supports' => '',
                'title'          => __( 'Top Header Section', 'wisdom-minimal' )
            )
        );

        /**
         * Checkbox for Top Header Section
         *
         * @since 1.0.0
         */
        $wp_customize->add_setting(
            'wisdom_minimal_top_header_option',
            array(
                'capability'        => 'edit_theme_options',
                'default'           => true,
                'sanitize_callback' => 'wisdom_blog_sanitize_checkbox',
            )
        );
        $wp_customize->add_control( new Wisdom_Blog_Toggle_Checkbox_Custom_Control(
            $wp_customize, 
            'wisdom_minimal_top_header_option',
                array(
                    'label'         => __( 'Top Header', 'wisdom-minimal' ),
                    'description'   => __( 'Option to control top header section.', 'wisdom-minimal' ),
                    'section'       => 'wisdom_minimal_top_header_section',
                    'settings'      => 'wisdom_minimal_top_header_option',
                    'priority'      => 5
                )
            )
        );
        
        /**
         * Text field for excerpt read more
         *
         * @since 1.0.0
         */
        $wp_customize->add_setting(
            'wisdom_minimal_excerpt_read_more',
            array(
                'capability'        => 'edit_theme_options',
                'default'           => __( 'Read More', 'wisdom-minimal' ),
                'sanitize_callback' => 'sanitize_text_field',
            )
        );
        $wp_customize->add_control(
            'wisdom_minimal_excerpt_read_more',
            array(
                'label'         => __( 'Read More Text', 'wisdom-minimal' ),
                'section'       => 'wisdom_blog_archive_section',
                'settings'      => 'wisdom_minimal_excerpt_read_more',
                'type'          => 'text',
                'priority'      => 25
            )
        );
    
        /**
         * Select field for pagination
         *
         * @since 1.0.0
         */
        $wp_customize->add_setting(
            'wisdom_minimal_archive_pagination',
            array(
                'capability'        => 'edit_theme_options',
                'default'           => 'default',
                'sanitize_callback' => 'wisdom_blog_sanitize_select',
            )
        );
        $wp_customize->add_control(
            'wisdom_minimal_archive_pagination',
            array(
                'label'         => __( 'Pagination Type', 'wisdom-minimal' ),
                'description'   => __( 'Choose types of pagination.', 'wisdom-minimal' ),
                'section'       => 'wisdom_blog_archive_section',
                'settings'      => 'wisdom_minimal_archive_pagination',
                'type'          => 'select',
                'priority'      => 30,
                'choices'       => array(
                    'default'   => __( 'Default', 'wisdom-minimal' ),
                    'numeric'   => __( 'Numeric', 'wisdom-minimal' )
                )
            )
        );

    }
endif;

add_action( 'customize_register', 'wisdom_minimal_customize_register', 20 );

/*-------------------------------------------------------------------------------------------------------------------------------*/
/**
 * Enqueue child theme styles and scripts
 */
add_action( 'wp_enqueue_scripts', 'wisdom_minimal_scripts', 20 );

function wisdom_minimal_scripts() {
    
    global $wisdom_minimal_version;
    
    wp_enqueue_style( 'wisdom-minimal-google-font', wisdom_minimal_fonts_url(), array(), null );
    
    wp_dequeue_style( 'wisdom-blog-style' );
    wp_dequeue_style( 'wisdom-blog-responsive-style' );
    
    wp_enqueue_style( 'wisdom-blog-parent-style', get_template_directory_uri() . '/style.css', array(), esc_attr( $wisdom_minimal_version ) );
    
    wp_enqueue_style( 'wisdom-blog-parent-responsive', get_template_directory_uri() . '/assets/css/cv-responsive.css', array(), esc_attr( $wisdom_minimal_version ) );
    
    wp_enqueue_style( 'wisdom-minimal', get_stylesheet_uri(), array(), esc_attr( $wisdom_minimal_version ) );
    
    $wisdom_minimal_theme_color = esc_attr( get_theme_mod( 'wisdom_blog_theme_color', '#f7b4b4' ) );
    
    $output_css = '';
    
    $output_css .= ".edit-link .post-edit-link,.reply .comment-reply-link,.widget_search .search-submit,.widget_search .search-submit,article.sticky:before,.widget_search .search-submit:hover,.cv-top-header-wrap{ background: ". esc_attr( $wisdom_minimal_theme_color ) ."}\n";
    
    $output_css .= "a,a:hover,a:focus,a:active,.entry-footer a:hover,.comment-author .fn .url:hover,.commentmetadata .comment-edit-link,#cancel-comment-reply-link,#cancel-comment-reply-link:before,.logged-in-as a,.widget a:hover,.widget a:hover::before,.widget li:hover::before,.banner-btn a:hover,.entry-title a:hover,.entry-title a:hover,.cat-links a:hover,.wisdom_blog_latest_posts .cv-post-title a:hover,#site-navigation ul li.current-menu-item > a, #site-navigation ul li:hover > a, #site-navigation ul li.current_page_ancestor > a, #site-navigation ul li.current_page_item > a{ color: ". esc_attr( $wisdom_minimal_theme_color ) ."}\n";

    $output_css .= "widget_search .search-submit,.widget_search .search-submit:hover,#secondary .widget .widget-title{ border-color: ". esc_attr( $wisdom_minimal_theme_color ) ."}\n";

    $refine_output_css = wisdom_blog_css_strip_whitespace( $output_css );

    wp_add_inline_style( 'wisdom-minimal', $refine_output_css );
    
}

/*-------------------------------------------------------------------------------------------------------------------------------*/

if( ! function_exists( 'wisdom_minimal_top_header_section' ) ) :
    
    /**
     * Function to display top header section
     */

    function wisdom_minimal_top_header_section() {

        $wisdom_minimal_top_header_option = get_theme_mod( 'wisdom_minimal_top_header_option', true );
        if( $wisdom_minimal_top_header_option === false ) {
            return;
        }

?>
        <div class="cv-top-header-wrap">
            <div class="cv-container">
                
                <div class="cv-top-left-section">
                    <div class="date-section"><?php echo esc_html( date_i18n('l, F d, Y') ) ;?></div>
                    <div class="cv-top-header-social"><?php wisdom_blog_social_media(); ?></div>
                </div><!-- .cv-top-left-section -->
                
                <div class="cv-top-right-section">
                    <nav id="top-navigation" class="top-navigation" role="navigation">
                        <?php wp_nav_menu( array( 'theme_location' => 'wisdom_minimal_top_menu', 'menu_id' => 'top-menu', 'depth' => 1, 'fallback_cb' => false ) );
                        ?>
                    </nav><!-- #site-navigation -->
                </div><!-- .cv-top-right-section -->

            </div><!-- .cv-container -->
        </div><!-- .cv-top-header-wrap -->
<?php
    }

endif;

add_action( 'wisdom_blog_before_header', 'wisdom_minimal_top_header_section', 10 );
