<?php
/**
 * Minipress functions and definitions
 *
 * @package Minipress
 * @since Minipress 1.0
 */

/**
 * Set the content width based on the theme's design and stylesheet.
 */
if ( ! isset( $content_width ) )
	$content_width = 500; /* pixels */

/**
 * Adjust the content width when the full width page template is being used
 */
function obox_set_content_width() {
	global $content_width;

	if ( is_page_template( 'full-width-page.php' ) )
		$content_width = 940;
}
add_action( 'template_redirect', 'obox_set_content_width' );

/*
 * Load Jetpack compatibility file.
 */
require( get_template_directory() . '/inc/jetpack.php' );

if ( ! function_exists( 'obox_setup' ) ):
/**
 * Sets up theme defaults and registers support for various WordPress features.
 */
function obox_setup() {
	/**
	 * Make theme available for translation
	 * Translations can be filed in the /languages/ directory
	 */
	load_theme_textdomain( 'obox', get_template_directory() . '/languages' );

	/**
	 * Add default posts and comments RSS feed links to head
	 */
	add_theme_support( 'automatic-feed-links' );

	/**
	 * Enable support for Post Thumbnails on posts and pages
	 *
	 * @link http://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
	 */
	add_theme_support( 'post-thumbnails' );
	add_image_size('500', 500, 9999);

	/**
	 * Enable theme support for Featured Content
	 */
	add_theme_support( 'featured-content', array(
		'featured_content_filter'	=> 'obox_get_featured_content',
		'max_posts'			=> 10,
	) );

	/**
	 * This theme uses wp_nav_menu() in one location.
	 */
	register_nav_menus( array(
		'primary' => __( 'Primary Menu', 'obox' )
	) );

	/**
	 * Add support for the Quote Post Formats
	 */
	add_theme_support( 'post-formats', array( 'quote') );

	/**
	 * Add support for custom backgrounds
	 */

	 add_theme_support( 'custom-background' );
}
endif; // obox_setup

/**
 * Tell WordPress to run obox_setup() when the 'after_setup_theme' hook is run.
 */
add_action( 'after_setup_theme', 'obox_setup' );

/**
 * Add support for custom header
 */
 require( get_template_directory() . '/inc/custom-header.php' );

/**
 * Enqueue scripts.
 */
function obox_scripts() {
	wp_enqueue_script( 'obox-navigation', get_template_directory_uri() . '/js/menu.js' , '', '', true);
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'obox_scripts' );

/**
 * Register the Minipress font files.
 */
function obox_register_fonts() {
	$protocol = is_ssl() ? 'https' : 'http';
	wp_register_style( 'obox-google-droid-sans', "$protocol://fonts.googleapis.com/css?family=Droid+Sans:400,700" );
	wp_register_style( 'obox-google-lobster', "$protocol://fonts.googleapis.com/css?family=Lobster" );
}
add_action( 'init', 'obox_register_fonts' );
/**
 * Enqueue main stylesheet & fonts on the front end.
 */
function obox_styles() {
	// Output the Google Fonts
	wp_enqueue_style( 'obox-google-droid-sans' );
	wp_enqueue_style( 'obox-google-lobster' );

	// Main stylesheet
	wp_enqueue_style( 'obox-style', get_stylesheet_uri() , array(), '1.0' );
}
add_action( 'wp_enqueue_scripts', 'obox_styles' );

/**
 * Enqueue font styles for Appearance -> Header.
 */
function obox_fonts_admin( $hook_suffix ) {
	if ( 'appearance_page_custom-header' != $hook_suffix )
		return;

	wp_enqueue_style( 'obox-google-droid-sans' );
	wp_enqueue_style( 'obox-google-lobster' );
}
add_action( 'admin_enqueue_scripts', 'obox_fonts_admin' );

/**
 * Dequeue font styles when Typekit is active
 */
function obox_dequeue_fonts() {
	/**
	 * We don't want to enqueue the font scripts if the blog
	 * has WP.com Custom Design and is using a 'Headings' font.
	 */
	if ( class_exists( 'TypekitData' ) && class_exists( 'CustomDesign' ) ) {
		if ( CustomDesign::is_upgrade_active() ) {
			$customfonts = TypekitData::get( 'families' );

			if ( ! $customfonts )
				return;

			$site_title = $customfonts[ 'site-title' ];
			$headings = $customfonts[ 'headings' ];
			$body_text = $customfonts[ 'body-text' ];

			if ( $site_title[ 'id' ] && $headings[ 'id' ] && $body_text[ 'id' ] ) :
				wp_dequeue_style( 'obox-google-droid-sans' );
				wp_dequeue_style( 'obox-google-lobster' );
			endif;

		}
	}
}
add_action( 'wp_enqueue_scripts', 'obox_dequeue_fonts' );

/**
 * Sets the post excerpt length to 40 words.
 */
function obox_excerpt_length( $length ) {
	return 40;
}
add_filter( 'excerpt_length', 'obox_excerpt_length' );

/**
 * Returns a "Continue Reading" link for excerpts
 */
function obox_continue_reading_link() {
	return ' <a class="more-link" href="'. esc_url( get_permalink() ) . '">' . __( 'Continue&nbsp;reading&nbsp;<span class="meta-nav">&rarr;</span>', 'obox' ) . '</a>';
}

/**
 * Replaces "[...]" (appended to automatically generated excerpts) with an ellipsis and obox_continue_reading_link().
 */
function obox_auto_excerpt_more( $more ) {
	return ' &hellip;' . obox_continue_reading_link();
}
add_filter( 'excerpt_more', 'obox_auto_excerpt_more' );

/**
 * Adds a pretty "Continue Reading" link to custom post excerpts.
 */
function obox_custom_excerpt_more( $output ) {
	if ( has_excerpt() && ! is_attachment() ) {
		$output .= obox_continue_reading_link();
	}
	return $output;
}
add_filter( 'get_the_excerpt', 'obox_custom_excerpt_more' );

/**
 * Get our wp_nav_menu() fallback, wp_page_menu(), to show a home link.
 */
function obox_page_menu_args( $args ) {
	$args['show_home'] = true;
	return $args;
}
add_filter( 'wp_page_menu_args', 'obox_page_menu_args' );

/**
 * Register widgetized area and update sidebar with default widgets
 */
function obox_widgets_init() {
	// Site-wide blog sidebar.
	register_sidebar( array(
		'name' => __( 'Sidebar' , 'obox' ),
		'id' => 'sidebar',
		'before_widget' => '<li id="%1$s" class="widget %2$s"><div class="content">',
		'after_widget' => '</div></li>',
		'before_title' => '<h4 class="widgettitle">',
		'after_title' => '</h4>',
	) );


}
add_action( 'widgets_init', 'obox_widgets_init' );

if ( ! function_exists( 'obox_content_nav' ) ):
/**
 * Display navigation to next/previous pages when applicable
 */
function obox_content_nav( $nav_id ) {
	global $wp_query; ?>
	<nav id="<?php echo $nav_id; ?>">
		<?php if ( $wp_query->max_num_pages > 1 && ( is_home() || is_archive() || is_search() ) ) : // navigation links for home, archive, and search pages ?>

			<?php if ( get_next_posts_link() ) : ?>
				<div class="nav-previous"><?php next_posts_link( __( '<span class="meta-nav">&larr;</span> Older posts', 'obox' ) ); ?></div>
			<?php endif; ?>

			<?php if ( get_previous_posts_link() ) : ?>
				<div class="nav-next"><?php previous_posts_link( __( 'Newer posts <span class="meta-nav">&rarr;</span>', 'obox' ) ); ?></div>
			<?php endif; ?>

		<?php endif; ?>
	</nav><!-- #<?php echo $nav_id; ?> -->
	<?php
}
endif; // obox_content_nav

/**
 * Filters wp_title to print a neat <title> tag based on what is being viewed.
 */
function obox_wp_title( $title, $sep ) {
	global $page, $paged;

	if ( is_feed() )
		return $title;

	// Add the blog name
	$title .= get_bloginfo( 'name' );

	// Add the blog description for the home/front page.
	$site_description = get_bloginfo( 'description', 'display' );
	if ( $site_description && ( is_home() || is_front_page() ) )
		$title .= " $sep $site_description";

	// Add a page number if necessary:
	if ( $paged >= 2 || $page >= 2 )
		$title .= " $sep " . sprintf( __( 'Page %s', '_s' ), max( $paged, $page ) );

	return $title;
}
add_filter( 'wp_title', 'obox_wp_title', 10, 2 );

/**
 * Let's make sure that the user's site looks respectable even without a menu setup
 */
function obox_menu_fallback(){
	echo '<ul id="nav" class="clearfix">';
		wp_list_pages( 'title_li=' );
	echo '</ul>';
}

if ( ! function_exists( 'obox_comment' ) ) :
/**
 * Template for comments and pingbacks.
 * Used as a callback by wp_list_comments() for displaying the comments.
 */
function obox_comment( $comment, $args, $depth ) {
	$GLOBALS['comment'] = $comment;
	switch ( $comment->comment_type ) :
		case 'pingback' :
		case 'trackback' :
	?>
	<li class="comment post pingback">
		<p><?php _e( 'Pingback:', 'obox' ); ?> <?php comment_author_link(); ?><?php edit_comment_link( __( '[Edit]', 'obox' ), ' ' ); ?></p>
	</li>
	<?php
			break;
		default :
	?>
	<li <?php comment_class(); ?> id="li-comment-<?php comment_ID(); ?>">
		<div class="comment-author vcard comment-avatar">
			<?php echo get_avatar($comment, $size = '35'); ?>
		</div>
		<div class="comment-meta commentmetadata comment-post clearfix">
			<?php if ($comment->comment_approved == '0') : ?>
				<em><?php _e('Your comment is awaiting moderation.', 'ocmx') ?></em>
				<br />
			<?php endif; ?>
			<h4 class="comment-name"><a href="<?php comment_author_url(); ?>" class="commentor_url" rel="nofollow"><?php comment_author(); ?></a></h4>
			<h5 class="date"><?php comment_date('d M Y h:m a') ?></h5>
			<?php comment_text() ?><?php edit_comment_link(__('(Edit)', 'ocmx'),'  ','') ?>
			<div class="reply reply-to-comment">
				<?php comment_reply_link(array_merge( $args, array('depth' => $depth, 'max_depth' => $args['max_depth']))) ?>
			</div>
		</div>
	</li><!-- #comment-## -->
	<?php
			break;
	endswitch;
}
endif; // ends check for obox_comment()

if ( ! function_exists( 'obox_posted_on' ) ) :


/**
 * Prints HTML with meta information for the current post-date/time and author.
 */
function obox_posted_on() {
	printf( __( '<span class="sep">Posted at </span> %1$s <span class="byline"> <span class="sep"> by </span> <span class="post-author vcard"><a class="url fn n" href="%2$s" title="%3$s" rel="author">%4$s</a></span></span>, on %5$s', 'obox' ),
		esc_attr( get_the_time() ),
		esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ),
		esc_attr( sprintf( __( 'View all posts by %s', 'obox' ), get_the_author() ) ),
		esc_attr( get_the_author() ),
		get_the_date( _x( 'F j, Y', 'month and yearly based time for archives and single posts', 'obox' ) )
	);
}
endif;

/**
 * Prints HTML with all the information related to the title in author.php
 */
function obox_author_title(){
	printf( __( 'Author Archives: %s', 'obox' ),
		'<span class="vcard"><a class="url fn n" href="' . esc_url( get_author_posts_url( get_the_author_meta( "ID" ) ) ) . '" title="' . esc_attr( get_the_author() ) . '" rel="me">' . get_the_author() . '</a></span>'
	);
}
/**
 * Adds custom classes to the array of body classes.
 */
function obox_body_classes( $classes ) {

	// Adds a class of single-author to blogs with only 1 published author
	if ( ! is_multi_author() ) {
		$classes[] = 'single-author';
	}

	// Adds a class of index to views that are not posts or pages or search
	if ( ! is_singular() && ! is_search() ) {
		$classes[] = 'index';
	}

	return $classes;
}
add_filter( 'body_class', 'obox_body_classes' );

/**
 * Returns true if a blog has more than 1 category
 */
function obox_categorized_blog() {
	if ( false === ( $all_the_cool_cats = get_transient( 'all_the_cool_cats' ) ) ) {
		// Create an array of all the categories that are attached to posts
		$all_the_cool_cats = get_categories( array(
			'hide_empty' => 1,
		) );

		// Count the number of categories that are attached to the posts
		$all_the_cool_cats = count( $all_the_cool_cats );

		set_transient( 'all_the_cool_cats', $all_the_cool_cats );
	}

	if ( '1' != $all_the_cool_cats ) {
		// This blog has more than 1 category so obox_categorized_blog should return true
		return true;
	} else {
		// This blog has only 1 category so obox_categorized_blog should return false
		return false;
	}
}

/**
 * Flush out the transients used in obox_categorized_blog
 */
function obox_category_transient_flusher() {
	// Like, beat it. Dig?
	delete_transient( 'all_the_cool_cats' );
}
add_action( 'edit_category', 'obox_category_transient_flusher' );
add_action( 'save_post', 'obox_category_transient_flusher' );


/**
 * Filter in a link to a content ID attribute for the next/previous image links on image attachment pages
 */
function obox_enhanced_image_navigation( $url = false, $id = false ) {
	if ( ! is_attachment() && ! wp_attachment_is_image( $id ) )
		return $url;

	$image = get_post( $id );
	if ( ! empty( $image->post_parent ) && $image->post_parent != $id )
		$url .= '#main';

	return $url;
}
add_filter( 'attachment_link', 'obox_enhanced_image_navigation' );

/**
 * Featured content
 */
function obox_get_featured_content() {
	return apply_filters( 'obox_get_featured_content', array() );
}

/**
 * Check for the presence of multiple Featured Content posts
 */
function obox_has_featured_posts( $minimum = 1 ) {
	if ( is_paged() )
		return false;

	$minimum = absint( $minimum );
	$featured_posts = apply_filters( 'obox_get_featured_content', array() );

	if ( ! is_array( $featured_posts ) )
		return false;

	if ( $minimum > count( $featured_posts ) )
		return false;

	return true;
}



/**
 * This theme was built with PHP, Semantic HTML, CSS, and love.
 */