<?php

/**
 * Magic Elementor functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Magic Elementor
 */


if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly.
}

if (!defined('MAGIC_ELEMENTOR_VERSION')) {
	define('MAGIC_ELEMENTOR_VERSION', '1.0.12');
}

// Define theme constants
define('MAGIC_ELEMENTOR_THEME_PATH', get_template_directory());
define('MAGIC_ELEMENTOR_THEME_URL', get_template_directory_uri());
define('MAGIC_ELEMENTOR_ASSETS_PATH', MAGIC_ELEMENTOR_THEME_PATH . '/assets/');
define('MAGIC_ELEMENTOR_ASSETS_URL', MAGIC_ELEMENTOR_THEME_URL . '/assets/');

if (!function_exists('magic_elementor_setup')) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function magic_elementor_setup()
	{
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on Magic Elementor, use a find and replace
		 * to change 'magic-elementor' to the name of your theme in all the template files.
		 */
		load_theme_textdomain('magic-elementor', get_template_directory() . '/languages');

		// Add default posts and comments RSS feed links to head.
		add_theme_support('automatic-feed-links');

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support('title-tag');

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support('post-thumbnails');


		// This theme uses wp_nav_menu() in one location.
		register_nav_menus(
			array(
				'main-menu' => esc_html__('Main Menu', 'magic-elementor'),
			)
		);

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support(
			'html5',
			array(
				'search-form',
				'comment-form',
				'comment-list',
				'gallery',
				'caption',
				'style',
				'script',
			)
		);

		// Set up the WordPress core custom background feature.
		add_theme_support(
			'custom-background',
			apply_filters(
				'magic_elementor_custom_background_args',
				array(
					'default-color' => 'ffffff',
					'default-image' => '',
				)
			)
		);

		// Add theme support for selective refresh for widgets.
		add_theme_support('customize-selective-refresh-widgets');
		// Add support for Block Styles.
		add_theme_support('wp-block-styles');

		// Add support for full and wide align images.
		add_theme_support('align-wide');
		add_theme_support("responsive-embeds");

		// Add support for editor styles
		add_theme_support('editor-styles');
		add_editor_style('assets/css/editor-styles.css');

		// Add support for custom spacing in the block editor
		add_theme_support('custom-spacing');

		// Add support for custom line height
		add_theme_support('custom-line-height');

		// Add support for custom units
		add_theme_support('custom-units');

		// Add support for link color in the block editor
		add_theme_support('link-color');

		/**
		 * Add support for core custom logo.
		 *
		 * @link https://codex.wordpress.org/Theme_Logo
		 */
		add_theme_support(
			'custom-logo',
			array(
				'height'      => 100,
				'width'       => 350,
				'flex-width'  => true,
				'flex-height' => true,
			)
		);

		/*
		 * WooCommerce support
		 */
		if (apply_filters('magic_elementor_add_woocommerce_support', true)) {
			// WooCommerce in general.
			add_theme_support('woocommerce');
			// Enabling WooCommerce product gallery features (are off by default since WC 3.0.0).
			// zoom.
			add_theme_support('wc-product-gallery-zoom');
			// lightbox.
			add_theme_support('wc-product-gallery-lightbox');
			// swipe.
			add_theme_support('wc-product-gallery-slider');
		}

		add_editor_style(array(magic_elementor_fonts_url()));

		$magicelementor_install_date = get_option('magicelementor_install_date');
		if (empty($magicelementor_install_date)) {
			update_option('magicelementor_install_date', current_time('mysql'));
		}
	}
endif;
add_action('after_setup_theme', 'magic_elementor_setup');



/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function magic_elementor_content_width()
{
	$GLOBALS['content_width'] = apply_filters('magic_elementor_content_width', 1170);
}
add_action('after_setup_theme', 'magic_elementor_content_width', 0);

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function magic_elementor_widgets_init()
{
	register_sidebar(
		array(
			'name'          => esc_html__('Sidebar', 'magic-elementor'),
			'id'            => 'sidebar-1',
			'description'   => esc_html__('Add widgets here.', 'magic-elementor'),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);
}
add_action('widgets_init', 'magic_elementor_widgets_init');

/**
 * Register custom fonts using Google Fonts API v2.
 */
function magic_elementor_fonts_url()
{
	$fonts_url = '';

	// Check if fonts are needed
	if (!apply_filters('magic_elementor_load_google_fonts', true)) {
		return '';
	}

	$font_families = array();

	// Google Fonts API v2 format
	$font_families[] = 'Poppins:ital,wght@0,400;0,700;1,400;1,700';
	$font_families[] = 'Roboto+Condensed:ital,wght@0,400;0,700;1,400;1,700';

	if (!empty($font_families)) {
		$query_args = array(
			'family' => implode('&family=', $font_families),
			'display' => 'swap', // Improves loading performance
		);

		$fonts_url = add_query_arg($query_args, 'https://fonts.googleapis.com/css2');
	}

	return esc_url_raw($fonts_url);
}


/**
 * Enqueue scripts and styles.
 */
function magic_elementor_scripts()
{
	wp_enqueue_style('magic-elementor-google-font', magic_elementor_fonts_url(), array(), null);
	wp_enqueue_style('magic-elementor-block-style', get_template_directory_uri() . '/assets/css/block.css', array(), MAGIC_ELEMENTOR_VERSION);
	wp_enqueue_style('magic-elementor-default-style', get_template_directory_uri() . '/assets/css/default-style.css', array(), MAGIC_ELEMENTOR_VERSION);
	wp_enqueue_style('magic-elementor-main-style', get_template_directory_uri() . '/assets/css/main.css', array(), MAGIC_ELEMENTOR_VERSION);
	wp_enqueue_style('magic-elementor-style', get_stylesheet_uri(), array(), MAGIC_ELEMENTOR_VERSION);
	wp_enqueue_style('magic-elementor-responsive-style', get_template_directory_uri() . '/assets/css/responsive.css', array(), MAGIC_ELEMENTOR_VERSION);

	wp_enqueue_script('magic-elementor-mobile-menu', get_template_directory_uri() . '/assets/js/mobile-menu.js', array(), MAGIC_ELEMENTOR_VERSION, true);
	wp_enqueue_script('magic-elementor-scripts', get_template_directory_uri() . '/assets/js/scripts.js', array('jquery'), MAGIC_ELEMENTOR_VERSION, true);

	if (is_singular() && comments_open() && get_option('thread_comments')) {
		wp_enqueue_script('comment-reply');
	}
}
add_action('wp_enqueue_scripts', 'magic_elementor_scripts');

function magic_elementor_gb_block_style()
{

	wp_enqueue_style('magic-elementor-gb-block', get_theme_file_uri('/assets/css/admin-block.css'), false, '1.0', 'all');
	wp_enqueue_style('magic-elementor-admin-google-font', magic_elementor_fonts_url(), array(), null);
}
add_action('enqueue_block_assets', 'magic_elementor_gb_block_style');

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer/customizer.php';


/**
 * Load Jetpack compatibility file.
 */
if (defined('JETPACK__VERSION')) {
	require get_template_directory() . '/inc/jetpack.php';
}

// Load all actions file
require get_template_directory() . '/actions/header-actions.php';

/**
 * Filter the except length to 20 words.
 *
 * @param int $length Excerpt length.
 * @return int (Maybe) modified excerpt length.
 */

add_filter('excerpt_more', '__return_empty_string', 999);
function newsx_paper_excerpt_length($length)
{
	return 30;
}
add_filter('excerpt_length', 'newsx_paper_excerpt_length', 999);



/**
 * Theme info
 */
require get_template_directory() . '/inc/theme-info.php';

/**
 * Add tem plugin activation
 */
require get_template_directory() . '/inc/class-tgm-plugin-activation.php';
require get_template_directory() . '/inc/recomended-plugin.php';

/**
 * Register Elementor Locations.
 */
if (!function_exists('magic_elementor_register_elementor_locations')) {
	function magic_elementor_register_elementor_locations($elementor_theme_manager)
	{
		if (apply_filters('magic_elementor_register_elementor_locations', true)) {
			$elementor_theme_manager->register_all_core_location();
		}
	}
}
add_action('elementor/theme/register_locations', 'magic_elementor_register_elementor_locations');

/**
 * Check whether to display header footer.
 */
if (!function_exists('magic_elementor_display_header_footer')) {
	function magic_elementor_display_header_footer()
	{
		$magic_elementor_header_footer = true;
		return apply_filters('magic_elementor_header_footer', $magic_elementor_header_footer);
	}
}

/**
 * Check whether to display the page title.
 */
if (!function_exists('magic_elementor_check_hide_title')) {
	function magic_elementor_check_hide_title($val)
	{
		if (defined('ELEMENTOR_VERSION')) {
			$current_doc = \Elementor\Plugin::instance()->documents->get(get_the_ID());
			if ($current_doc && 'yes' === $current_doc->get_settings('hide_title')) {
				$val = false;
			}
		}
		return $val;
	}
}
add_filter('magic_elementor_page_title', 'magic_elementor_check_hide_title');

/**
 * Add description meta tag with excerpt text.
 */
if (!function_exists('magic_elementor_add_description_meta_tag')) {
	function magic_elementor_add_description_meta_tag()
	{
		if (!apply_filters('magic_elementor_description_meta_tag', true)) {
			return;
		}

		if (!is_singular()) {
			return;
		}

		$post = get_queried_object();
		if (empty($post->post_excerpt)) {
			return;
		}

		echo '<meta name="description" content="' . esc_attr(wp_strip_all_tags($post->post_excerpt)) . '">' . "\n";
	}
}
add_action('wp_head', 'magic_elementor_add_description_meta_tag');
