<?php
/**
 * Functions which enhance the theme by hooking into WordPress
 *
 * @package jalbee
 */

/**
 * Adds custom classes to the array of body classes.
 *
 * @param array $classes Classes for the body element.
 * @return array
 */
function jalbee_body_classes( $classes ) {
	// Adds a class of hfeed to non-singular pages.
	if ( ! is_singular() ) {
		$classes[] = 'hfeed';
	}

	if ( is_active_sidebar( 'sidebar-1' ) ) {

		//Over ride sidebar position if post metabox set.
		$jalbee_sidebar_position = get_post_meta( get_the_ID(), 'sidebar_position_field', true );
		
		if ( ( get_theme_mod( 'sidebar_switch' ) || 'left_side' == $jalbee_sidebar_position ) && 'right_side' !== $jalbee_sidebar_position ) {
			$classes[] = 'right-sidebar';
		}

		$hide_sidebar = get_theme_mod( 'hide_sidebar' );
		global $template;
		if ( ( 'front-page.php' !== basename( $template ) ) && ! $hide_sidebar ) {

			$hide_sidebar_page    = ( is_page() && get_theme_mod( 'hide_sidebar_page' ) );
			$hide_sidebar_single  = ( is_single() && get_theme_mod( 'hide_sidebar_single' ) );
			$hide_sidebar_archive = ( is_archive() && get_theme_mod( 'hide_sidebar_archive' ) );
			$hide_sidebar_home    = ( is_home() && get_theme_mod( 'hide_sidebar_home', true ) );

			if ( $hide_sidebar_page || $hide_sidebar_single || $hide_sidebar_archive || $hide_sidebar_home ) {
				return $classes;
			}
			//Check metabox sidebar display settings.
			if( is_singular() && 'none' == $jalbee_sidebar_position ){
				return $classes;
			}

			$classes[] = 'has-sidebar';
		}
	}

	return $classes;
}
add_filter( 'body_class', 'jalbee_body_classes' );

/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 */
function jalbee_pingback_header() {
	if ( is_singular() && pings_open() ) {
		echo '<link rel="pingback" href="', esc_url( get_bloginfo( 'pingback_url' ) ), '">';
	}
}
add_action( 'wp_head', 'jalbee_pingback_header' );

/**
 * Adds async/defer attributes to enqueued / registered scripts.
 *
 * If #12009 lands in WordPress, this function can no-op since it would be handled in core.
 *
 * @link https://core.trac.wordpress.org/ticket/12009
 * @param string $tag    The script tag.
 * @param string $handle The script handle.
 * @return array
 */
function jalbee_filter_script_loader_tag( $tag, $handle ) {

	foreach ( array( 'async', 'defer' ) as $attr ) {
		if ( ! wp_scripts()->get_data( $handle, $attr ) ) {
			continue;
		}

		// Prevent adding attribute when already added in #12009.
		if ( ! preg_match( ":\s$attr(=|>|\s):", $tag ) ) {
			$tag = preg_replace( ':(?=></script>):', " $attr", $tag, 1 );
		}

		// Only allow async or defer, not both.
		break;
	}

	return $tag;
}

add_filter( 'script_loader_tag', 'jalbee_filter_script_loader_tag', 10, 2 );

/**
 * Generate preload markup for stylesheets.
 *
 * @param object $wp_styles Registered styles.
 * @param string $handle The style handle.
 */
function jalbee_get_preload_stylesheet_uri( $wp_styles, $handle ) {
	$preload_uri = $wp_styles->registered[ $handle ]->src . '?ver=' . $wp_styles->registered[ $handle ]->ver;
	return $preload_uri;
}

/**
 * Adds preload for in-body stylesheets depending on what templates are being used.
 * Disabled when AMP is active as AMP injects the stylesheets inline.
 *
 * @link https://developer.mozilla.org/en-US/docs/Web/HTML/Preloading_content
 */
function jalbee_add_body_style() {

	// If AMP is active, do nothing.
	if ( jalbee_is_amp() ) {
		return;
	}

	// Get registered styles.
	$wp_styles = wp_styles();

	$preloads = array();

	// Preload content.css.
	$preloads['jalbee-content'] = jalbee_get_preload_stylesheet_uri( $wp_styles, 'jalbee-content' );

	// Preload sidebar.css and widget.css.
	if ( is_active_sidebar( 'sidebar-1' ) ) {
		$preloads['jalbee-sidebar'] = jalbee_get_preload_stylesheet_uri( $wp_styles, 'jalbee-sidebar' );
		$preloads['jalbee-widgets'] = jalbee_get_preload_stylesheet_uri( $wp_styles, 'jalbee-widgets' );
	}

	// Preload comments.css.
	if ( ! post_password_required() && is_singular() && ( comments_open() || get_comments_number() ) ) {
		$preloads['jalbee-comments'] = jalbee_get_preload_stylesheet_uri( $wp_styles, 'jalbee-comments' );
	}

	// Preload front-page.css.
	global $template;
	if ( 'front-page.php' === basename( $template ) ) {
		$preloads['jalbee-front-page'] = jalbee_get_preload_stylesheet_uri( $wp_styles, 'jalbee-front-page' );
	}

	// Output the preload markup in <head>.
	foreach ( $preloads as $handle => $src ) {
		echo '<link rel="preload" id="' . esc_attr( $handle ) . '-preload" href="' . esc_url( $src ) . '" as="style" />';
		echo "\n";
	}

}
add_action( 'wp_head', 'jalbee_add_body_style' );

/**
 * Add dropdown symbol to nav menu items with children.
 *
 * Adds the dropdown markup after the menu link element,
 * before the submenu.
 *
 * Javascript converts the symbol to a toggle button.
 *
 * @TODO:
 * - This doesn't work for the page menu because it
 *   doesn't have a similar filter. So the dropdown symbol
 *   is only being added for page menus if JS is enabled.
 *   Create a ticket to add to core?
 *
 * @param string   $item_output The menu item's starting HTML output.
 * @param WP_Post  $item        Menu item data object.
 * @param int      $depth       Depth of menu item. Used for padding.
 * @param stdClass $args        An object of wp_nav_menu() arguments.
 * @return string Modified nav menu HTML.
 */
function jalbee_add_primary_menu_dropdown_symbol( $item_output, $item, $depth, $args ) {

	// Only for our primary menu location.
	if ( empty( $args->theme_location ) || 'primary' != $args->theme_location ) {
		return $item_output;
	}

	// Add the dropdown for items that have children.
	if ( ! empty( $item->classes ) && in_array( 'menu-item-has-children', $item->classes ) ) {
		return $item_output . '<span class="dropdown"><i class="dropdown-symbol"></i></span>';
	}

	return $item_output;
}
add_filter( 'walker_nav_menu_start_el', 'jalbee_add_primary_menu_dropdown_symbol', 10, 4 );

/**
 * Filters the HTML attributes applied to a menu item's anchor element.
 *
 * Checks if the menu item is the current menu
 * item and adds the aria "current" attribute.
 *
 * @param array   $atts   The HTML attributes applied to the menu item's `<a>` element.
 * @param WP_Post $item  The current menu item.
 * @return array Modified HTML attributes
 */
function jalbee_add_nav_menu_aria_current( $atts, $item ) {
	/*
	 * First, check if "current" is set,
	 * which means the item is a nav menu item.
	 *
	 * Otherwise, it's a post item so check
	 * if the item is the current post.
	 */
	if ( isset( $item->current ) ) {
		if ( $item->current ) {
			$atts['aria-current'] = 'page';
		}
	} else if ( ! empty( $item->ID ) ) {
		global $post;
		if ( ! empty( $post->ID ) && $post->ID == $item->ID ) {
			$atts['aria-current'] = 'page';
		}
	}

	return $atts;
}
add_filter( 'nav_menu_link_attributes', 'jalbee_add_nav_menu_aria_current', 10, 2 );
add_filter( 'page_menu_link_attributes', 'jalbee_add_nav_menu_aria_current', 10, 2 );
