<?php
/**
 * Jalbee functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package jalbee
 */

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function jalbee_setup() {
	/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on jalbee, use a find and replace
		* to change 'jalbee' to the name of your theme in all the template files.
		*/
	load_theme_textdomain( 'jalbee', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
	add_theme_support( 'title-tag' );

	/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus(
		array(
			'primary' => esc_html__( 'Primary', 'jalbee' ),
		)
	);

	/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		)
	);

	// Set up the WordPress core custom background feature.
	add_theme_support(
		'custom-background',
		apply_filters(
			'jalbee_custom_background_args',
			array(
				'default-color' => 'ffffff',
				'default-image' => '',
			)
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	/**
	 * Add support for default block styles.
	 *
	 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/#default-block-styles
	 */
	add_theme_support( 'wp-block-styles' );
	/**
	 * Add support for wide aligments.
	 *
	 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/#wide-alignment
	 */
	add_theme_support( 'align-wide' );

	/**
	 * Add support for block color palettes.
	 *
	 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/#block-color-palettes
	 */
	add_theme_support(
		'editor-color-palette',
		array(
			array(
				'name'  => __( 'Dusty orange', 'jalbee' ),
				'slug'  => 'dusty-orange',
				'color' => '#ed8f5b',
			),
			array(
				'name'  => __( 'Dusty red', 'jalbee' ),
				'slug'  => 'dusty-red',
				'color' => '#e36d60',
			),
			array(
				'name'  => __( 'Dusty wine', 'jalbee' ),
				'slug'  => 'dusty-wine',
				'color' => '#9c4368',
			),
			array(
				'name'  => __( 'Dark sunset', 'jalbee' ),
				'slug'  => 'dark-sunset',
				'color' => '#33223b',
			),
			array(
				'name'  => __( 'Almost black', 'jalbee' ),
				'slug'  => 'almost-black',
				'color' => '#0a1c28',
			),
			array(
				'name'  => __( 'Dusty water', 'jalbee' ),
				'slug'  => 'dusty-water',
				'color' => '#41848f',
			),
			array(
				'name'  => __( 'Dusty sky', 'jalbee' ),
				'slug'  => 'dusty-sky',
				'color' => '#72a7a3',
			),
			array(
				'name'  => __( 'Dusty daylight', 'jalbee' ),
				'slug'  => 'dusty-daylight',
				'color' => '#97c0b7',
			),
			array(
				'name'  => __( 'Dusty sun', 'jalbee' ),
				'slug'  => 'dusty-sun',
				'color' => '#eee9d1',
			),
		)
	);

	/**
	 * Optional: Disable custom colors in block color palettes.
	 *
	 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/
	 *
	 * add_theme_support( 'disable-custom-colors' );
	 */

	/**
	 * Add support for font sizes.
	 *
	 * @link https://wordpress.org/gutenberg/handbook/extensibility/theme-support/#block-font-sizes
	 */
	add_theme_support(
		'editor-font-sizes',
		array(
			array(
				'name'      => __( 'small', 'jalbee' ),
				'shortName' => __( 'S', 'jalbee' ),
				'size'      => 16,
				'slug'      => 'small',
			),
			array(
				'name'      => __( 'regular', 'jalbee' ),
				'shortName' => __( 'M', 'jalbee' ),
				'size'      => 20,
				'slug'      => 'regular',
			),
			array(
				'name'      => __( 'large', 'jalbee' ),
				'shortName' => __( 'L', 'jalbee' ),
				'size'      => 36,
				'slug'      => 'large',
			),
			array(
				'name'      => __( 'larger', 'jalbee' ),
				'shortName' => __( 'XL', 'jalbee' ),
				'size'      => 48,
				'slug'      => 'larger',
			),
		)
	);

	/**
	 * Optional: Add AMP support.
	 *
	 * Add built-in support for the AMP plugin and specific AMP features.
	 * Control how the plugin, when activated, impacts the theme.
	 *
	 * @link https://wordpress.org/plugins/amp/
	 */
	add_theme_support(
		'amp',
		array(
			'comments_live_list' => true,
		)
	);

}
add_action( 'after_setup_theme', 'jalbee_setup' );

/**
 * Set the embed width in pixels, based on the theme's design and stylesheet.
 *
 * @param array $dimensions An array of embed width and height values in pixels (in that order).
 * @return array
 */
function jalbee_embed_dimensions( array $dimensions ) {
	$dimensions['width'] = 720;
	return $dimensions;
}
add_filter( 'embed_defaults', 'jalbee_embed_dimensions' );

/**
 * Register Google Fonts
 */
function jalbee_fonts_url() {
	$fonts_url = '';

	// font-family: 'Open Sans', sans-serif;
	// font-family: 'Montserrat', sans-serif;
	/**
	 * Translator: If Montserrat does not support characters in your language, translate this to 'off'.
	 */
	$montserrat = esc_html_x( 'on', 'Montserrat font: on or off', 'jalbee' );
	/**
	 * Translator: If Open Sans  does not support characters in your language, translate this to 'off'.
	 */
	$open_sans = esc_html_x( 'on', 'Open Sans font: on or off', 'jalbee' );

	$font_families = array();

	if ( 'off' !== $montserrat ) {
		$font_families[] = 'Montserrat:400,500,700';
	}

	if ( 'off' !== $open_sans ) {
		$font_families[] = 'Open+Sans:300,400,600';
	}

	if ( in_array( 'on', array( $open_sans, $montserrat ) ) ) {
		$query_args = array(
			'family' => urlencode( implode( '|', $font_families ) ),
			'subset' => urlencode( 'latin,latin-ext' ),
		);

		$fonts_url = add_query_arg( $query_args, 'https://fonts.googleapis.com/css' );
	}

	return esc_url_raw( $fonts_url );

}

/**
 * Add preconnect for Google Fonts.
 *
 * @since Twenty Seventeen 1.0
 *
 * @param array  $urls           URLs to print for resource hints.
 * @param string $relation_type  The relation type the URLs are printed.
 * @return array $urls           URLs to print for resource hints.
 */
function jalbee_resource_hints( $urls, $relation_type ) {
	if ( wp_style_is( 'jalbee-fonts', 'queue' ) && 'preconnect' === $relation_type ) {
		$urls[] = array(
			'href' => 'https://fonts.gstatic.com',
			'crossorigin',
		);
	}

	return $urls;
}
add_filter( 'wp_resource_hints', 'jalbee_resource_hints', 10, 2 );

/**
 * Enqueue WordPress theme styles within Gutenberg.
 */
function jalbee_gutenberg_styles() {
	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'jalbee-fonts', jalbee_fonts_url(), array(), null ); // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion

	// Enqueue main stylesheet.
	wp_enqueue_style( 'jalbee-base-style', get_theme_file_uri( '/css/editor-styles.css' ), array(), '20180514' );
}
add_action( 'enqueue_block_editor_assets', 'jalbee_gutenberg_styles' );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function jalbee_widgets_init() {
	register_sidebar(
		array(
			'name'          => esc_html__( 'Sidebar', 'jalbee' ),
			'id'            => 'sidebar-1',
			'description'   => esc_html__( 'Add widgets here.', 'jalbee' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);

		register_sidebar(
			array(
				'name'          => esc_html__( 'Footer Area', 'jalbee' ),
				'id'            => 'footer-widget',
				'description'   => esc_html__( 'Add widgets here.', 'jalbee' ),
				'before_widget' => '<div id="%1$s" class="footer-widget widget %2$s">',
				'after_widget'  => '</div>',
				'before_title'  => '<h3 class="widget-title">',
				'after_title'   => '</h3>',
			)
		);

}
add_action( 'widgets_init', 'jalbee_widgets_init' );

/**
 * Enqueue styles.
 */
function jalbee_styles() {
	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'jalbee-fonts', jalbee_fonts_url(), array(), null ); // phpcs:ignore WordPress.WP.EnqueuedResourceParameters.MissingVersion

	// Enqueue main stylesheet.
	wp_enqueue_style( 'jalbee-base-style', get_stylesheet_uri(), array(), '20180514' );

	// Register component styles that are printed as needed.
	wp_register_style( 'jalbee-comments', get_theme_file_uri( '/css/comments.css' ), array(), '20180514' );
	wp_register_style( 'jalbee-hero', get_theme_file_uri( '/css/hero.css' ), array(), '20180514' );
	wp_register_style( 'jalbee-content', get_theme_file_uri( '/css/content.css' ), array(), '20180514' );
	wp_register_style( 'jalbee-sidebar', get_theme_file_uri( '/css/sidebar.css' ), array(), '20180514' );
	wp_register_style( 'jalbee-widgets', get_theme_file_uri( '/css/widgets.css' ), array(), '20180514' );
	wp_register_style( 'jalbee-front-page', get_theme_file_uri( '/css/front-page.css' ), array(), '20180514' );
}
add_action( 'wp_enqueue_scripts', 'jalbee_styles' );

/**
 * Enqueue scripts.
 */
function jalbee_scripts() {

	// If the AMP plugin is active, return early.
	if ( jalbee_is_amp() ) {
		return;
	}

	// Enqueue the navigation script.
	wp_enqueue_script( 'jalbee-main', get_theme_file_uri( '/js/main.js' ), array( 'jquery' ), '20180514', false );
	wp_localize_script(
		'jalbee-main',
		'jalbee',
		array(
			'volume_mute_image' => get_theme_file_uri( '/images/volume-mute.svg' ),
			'volume_up_image'   => get_theme_file_uri( '/images/volume-up.svg' ),
		)
	);
	wp_script_add_data( 'jalbee-main', 'async', true );

	wp_enqueue_script( 'jalbee-navigation', get_theme_file_uri( '/js/navigation.js' ), array(), '20180514', false );
	wp_script_add_data( 'jalbee-navigation', 'async', true );
	wp_localize_script(
		'jalbee-navigation',
		'jalbeeScreenReaderText',
		array(
			'expand'   => __( 'Expand child menu', 'jalbee' ),
			'collapse' => __( 'Collapse child menu', 'jalbee' ),
		)
	);

	/**
	 * Load Custom Customizer CSS
	 */
	function jalbee_customize_preview_css() {
		wp_enqueue_style( 'jalbee_customizer_style', get_theme_file_uri( '/css/customizer.css' ) );
	}

	add_action( 'customize_controls_enqueue_scripts', 'jalbee_customize_preview_css', 11 );

	// Enqueue skip-link-focus script.
	wp_enqueue_script( 'jalbee-skip-link-focus-fix', get_theme_file_uri( '/js/skip-link-focus-fix.js' ), array(), '20180514', false );
	wp_script_add_data( 'jalbee-skip-link-focus-fix', 'defer', true );

	// Enqueue comment script on singular post/page views only.
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

}
add_action( 'wp_enqueue_scripts', 'jalbee_scripts' );


/**
 * If Amp not activated.
 *
 * @return void
 */
function jalbee_is_amp_activated() {
	if ( ! jalbee_is_amp() ) : ?>
		<script>document.documentElement.classList.remove("no-js");</script>
	<?php
	endif; 
}
add_action( 'wp_footer', 'jalbee_is_amp_activated' );

/**
 * Customizer css.
 */
function jalbee_customize_css() {

	require get_template_directory() . '/inc/custom-css.php';

}
add_action( 'wp_head', 'jalbee_customize_css' );

/**
 * Custom responsive image sizes.
 */
require get_template_directory() . '/inc/image-sizes.php';


/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Meta box for styling.
 */
require get_template_directory() . '/inc/meta-box.php';

/**
 * Optional: Add theme support for lazyloading images.
 *
 * @link https://developers.google.com/web/fundamentals/performance/lazy-loading-guidance/images-and-video/
 */
require get_template_directory() . '/pluggable/lazyload/lazyload.php';

/**
 * Recomended plugins.
 */
require get_template_directory() . '/inc/tgma-plugins.php';


