<?php
/**
 * Featured Category Component
 * 
 * Displays 3 latest posts from a selected category in a grid layout
 * 
 * @package Graceful Glow Blog
 * @since 1.0.0
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Display Featured Category Section
 */
function featured_category_display() {
	// Check if section is enabled
	$section_enabled = get_theme_mod( 'featured_category_enabled', false );
	if ( ! $section_enabled ) {
		return;
	}
	
	// Get customizer settings
	$selected_category = absint( get_theme_mod( 'featured_category_select', 0 ) );
	$section_title = sanitize_text_field( get_theme_mod( 'featured_category_title', esc_html__( 'Featured Posts', 'graceful-glow-blog' ) ) );
	$badge_color = sanitize_hex_color( get_theme_mod( 'featured_category_badge_color', '#f59e0b' ) );
	
	// If no category selected, return
	if ( $selected_category === 0 ) {
		return;
	}
	
	// Query 3 latest posts from selected category
	$query_args = array(
		'posts_per_page' => 3,
		'post_status' => 'publish',
		'cat' => $selected_category,
		'meta_query' => array(
			array(
				'key' => '_thumbnail_id',
				'compare' => 'EXISTS',
			),
		),
		'orderby' => 'date',
		'order' => 'DESC',
		'no_found_rows' => true,
		'update_post_meta_cache' => false,
		'update_post_term_cache' => false,
	);
	
	$posts_query = new WP_Query( $query_args );
	
	// Check for errors
	if ( is_wp_error( $posts_query ) ) {
		return;
	}
	
	// Handle empty results
	if ( ! $posts_query->have_posts() ) {
		wp_reset_postdata();
		return;
	}
	
	// Prepare posts data
	$posts_data = array();
	
	while ( $posts_query->have_posts() ) {
		$posts_query->the_post();
		
		// Get featured image
		$featured_image = get_the_post_thumbnail_url( get_the_ID(), 'large' );
		
		if ( ! $featured_image ) {
			$featured_image = get_the_post_thumbnail_url( get_the_ID(), 'medium_large' );
		}
		
		if ( ! $featured_image ) {
			continue;
		}
		
		$post_title = get_the_title();
		$post_permalink = get_permalink();
		
		if ( empty( $post_title ) || empty( $post_permalink ) ) {
			continue;
		}
		
		// Get first category
		$categories = get_the_category();
		$category_name = '';
		if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
			$category_name = esc_html( strtoupper( $categories[0]->name ) );
		}
		
		// Get post date
		$post_date = get_the_date( 'F j, Y' );
		
		$posts_data[] = array(
			'id' => get_the_ID(),
			'title' => $post_title,
			'permalink' => $post_permalink,
			'featured_image' => $featured_image,
			'category' => $category_name,
			'date' => $post_date,
		);
	}
	
	wp_reset_postdata();
	
	if ( empty( $posts_data ) ) {
		return;
	}
	
	// Render HTML
	?>
	<style>
		.featured-category-badge {
			background: <?php echo esc_attr( $badge_color ); ?> !important;
		}
	</style>
	<div class="featured-category-section">
		<?php if ( ! empty( $section_title ) ) : ?>
			<h2 class="featured-category-title"><?php echo esc_html( $section_title ); ?></h2>
		<?php endif; ?>
		
		<div class="featured-category-grid">
			<?php foreach ( $posts_data as $post ) : ?>
				<article class="featured-category-card">
					<a href="<?php echo esc_url( $post['permalink'] ); ?>" class="featured-category-link">
						<div class="featured-category-image" style="background-image: url('<?php echo esc_url( $post['featured_image'] ); ?>');">
							<div class="featured-category-overlay"></div>
							<div class="featured-category-content">
								<?php if ( ! empty( $post['category'] ) ) : ?>
									<span class="featured-category-badge"><?php echo esc_html( $post['category'] ); ?></span>
								<?php endif; ?>
								<span class="featured-category-date"><?php echo esc_html( strtoupper( $post['date'] ) ); ?></span>
								<h3 class="featured-category-post-title"><?php echo esc_html( $post['title'] ); ?></h3>
							</div>
						</div>
					</a>
				</article>
			<?php endforeach; ?>
		</div>
	</div>
	<?php
}
