<?php
/**
 * Custom theme functions.
 *
 * This file contains hook functions attached to theme hooks.
 *
 * @package Echoes
 */

if ( ! function_exists( 'echoes_skip_to_content' ) ) :

	/**
	 * Add Skip to content.
	 *
	 * @since 1.0
	 */
	function echoes_skip_to_content() {
	?><a class="skip-link screen-reader-text" href="#content"><?php esc_html_e( 'Skip to content', 'echoes' ); ?></a><?php
	}

endif;

add_action( 'echoes_action_before', 'echoes_skip_to_content', 15 );

if ( ! function_exists( 'echoes_site_branding' ) ) :

	/**
	 * Site branding.
	 *
	 * @since 1.0
	 */
	function echoes_site_branding() {

	?>
	
    <div class="site-branding">
		<?php echoes_the_custom_logo(); ?>
		<?php $show_title = echoes_get_option( 'show_title' ); ?>
		<?php $show_tagline = echoes_get_option( 'show_tagline' ); ?>
		<?php if ( true === $show_title || true === $show_tagline ) : ?>
			<div id="site-header">
				<?php if ( true === $show_title ) :  ?>
					<?php if ( is_front_page() && is_home() ) : ?>
						<h1 class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" class="navbar-brand" rel="home"><?php bloginfo( 'name' ); ?></a></h1>
					<?php else : ?>
						<p class="site-title"><a href="<?php echo esc_url( home_url( '/' ) ); ?>" class="navbar-brand" rel="home"><?php bloginfo( 'name' ); ?></a></p>
					<?php endif; ?>
				<?php endif; ?>

				<?php if ( true === $show_tagline ) : ?>
					<p class="site-description"><?php bloginfo( 'description' ); ?></p>
				<?php endif; ?>
			</div><!-- #site-identity -->
		<?php endif; ?>
    </div><!-- .site-branding -->

    <?php

	}

endif;

add_action( 'echoes_action_header', 'echoes_site_branding' );

if ( ! function_exists( 'echoes_add_primary_navigation' ) ) :

	/**
	 * Site branding.
	 *
	 * @since 1.0
	 */
	function echoes_add_primary_navigation() {
	?>
    <nav id="site-navigation" class="main-navigation" role="navigation">
    	<button type="button" class="menu-toggle">
            <span class="icon-bar"></span>
            <span class="icon-bar"></span>
            <span class="icon-bar"></span>
        </button>
        
		<?php
		wp_nav_menu( array(
			'theme_location' => 'primary',
			'menu_id'        => 'primary-menu',
			'menu_class'     => 'nav-menu',
			'fallback_cb'    => 'echoes_primary_navigation_fallback',
		) );
		?>
    </nav><!-- #site-navigation -->
    <?php
	}

endif;
add_action( 'echoes_action_header', 'echoes_add_primary_navigation', 20 );

if ( ! function_exists( 'echoes_footer_copyright' ) ) :

	/**
	 * Footer copyright
	 *
	 * @since 1.0
	 */
	function echoes_footer_copyright() {

		// Check if footer is disabled.
		$footer_status = apply_filters( 'echoes_filter_footer_status', true );
		if ( true !== $footer_status ) {
			return;
		}

		// Copyright content.
		$copyright_text = echoes_get_option( 'copyright_text' );
		$copyright_text = apply_filters( 'echoes_filter_copyright_text', $copyright_text );
		if ( ! empty( $copyright_text ) ) {
			$copyright_text = wp_kses_data( $copyright_text );
		}

	?>

    <?php if ( ! empty( $footer_menu_content ) ) :  ?>
		<?php echo wp_kses_post( $footer_menu_content ); ?>
    <?php endif ?>

    	
    <div class="site-info">
    	<div class="container">
			<p><?php echo wp_kses_data( $copyright_text ); ?><?php printf( esc_html__( '| Proudly Powered by %1$s and %2$s', 'echoes' ), '<a target="_blank" href="' . esc_url( 'https://wordpress.org/' ) . '">WordPress</a>', '<a target="_blank" href="' . esc_url( 'https://wordpress.org/themes/echoes/' ) . '">Echoes</a>' ); ?></a></p>
    	</div><!-- .container -->
    </div><!-- .site-info -->
        
    <?php
	}

endif;

add_action( 'echoes_action_footer', 'echoes_footer_copyright', 10 );


if ( ! function_exists( 'echoes_add_sidebar' ) ) :

	/**
	 * Add sidebar.
	 *
	 * @since 1.0
	 */
	function echoes_add_sidebar() {

		global $post;

		$page_layout = echoes_get_option( 'page_layout' );
		$page_layout = apply_filters( 'echoes_filter_theme_page_layout', $page_layout );

		// Check if single.
		if ( $post && is_singular() ) {
			$post_options = get_post_meta( $post->ID, 'echoes_theme_settings', true );
			if ( isset( $post_options['post_layout'] ) && ! empty( $post_options['post_layout'] ) ) {
				$page_layout = $post_options['post_layout'];
			}
		}

		// Include primary sidebar.
		if ( 'no-sidebar' !== $page_layout ) {
			get_sidebar();
		}
	}

endif;

add_action( 'echoes_action_sidebar', 'echoes_add_sidebar' );


if ( ! function_exists( 'echoes_custom_posts_navigation' ) ) :
	/**
	 * Posts navigation.
	 *
	 * @since 1.0
	 */
	function echoes_custom_posts_navigation() {

		$pagination_type = echoes_get_option( 'pagination_type' );

		switch ( $pagination_type ) {

			case 'default':
				the_posts_navigation();
			break;

			case 'numeric':
				the_posts_pagination();
			break;

			default:
			break;
		}

	}
endif;

add_action( 'echoes_action_posts_navigation', 'echoes_custom_posts_navigation' );


if ( ! function_exists( 'echoes_add_image_in_single_display' ) ) :

	/**
	 * Add image in single post.
	 *
	 * @since 1.0
	 */
	function echoes_add_image_in_single_display() {

		global $post;

		// Bail if checkbox Use Featured Image as Banner is enabled.
		$values = get_post_meta( $post->ID, 'echoes_theme_settings', true );
        if ( isset( $values['use_featured_image_as_banner'] ) && 1 === absint( $values['use_featured_image_as_banner'] ) ) {
			return;
        }
		if ( has_post_thumbnail() ) {

			$values = get_post_meta( $post->ID, 'echoes_theme_settings', true );
			$single_image = isset( $values['single_image'] ) ? esc_attr( $values['single_image'] ) : '';

			if ( ! $single_image ) {
				$single_image = echoes_get_option( 'single_image' );
			}

			if ( 'disable' !== $single_image ) {
				$args = array(
					'class' => 'aligncenter',
				);
				the_post_thumbnail( esc_attr( $single_image ), $args );
			}
		}

	}

endif;

add_action( 'echoes_single_image', 'echoes_add_image_in_single_display' );

if( ! function_exists( 'echoes_check_custom_header_status' ) ) :

	/**
	 * Check status of custom header.
	 *
	 * @since 1.0
	 */
	function echoes_check_custom_header_status( $input ) {

		global $post;

		if ( is_front_page() && 'posts' === get_option( 'show_on_front' ) ) {
			$input = false;
		}
		else if ( is_home() && ( $blog_page_id = echoes_get_index_page_id( 'blog' ) ) > 0 ) {
			$values = get_post_meta( $blog_page_id, 'echoes_theme_settings', true );
			$disable_banner_area = isset( $values['disable_banner_area'] ) ? absint( $values['disable_banner_area'] ) : 0;
			if ( 1 === $disable_banner_area ) {
				$input = false;
			}
		}
		else if ( $post ) {
			if ( is_singular() ) {
				$values = get_post_meta( $post->ID, 'echoes_theme_settings', true );
				$disable_banner_area = isset( $values['disable_banner_area'] ) ? absint( $values['disable_banner_area'] ) : 0;
				if ( 1 === $disable_banner_area ) {
					$input = false;
				}
			}
		}

		return $input;

	}

endif;

add_filter( 'echoes_filter_custom_header_status', 'echoes_check_custom_header_status' );

if ( ! function_exists( 'echoes_customize_page_title' ) ) :

	/**
	 * Add title in Custom Header.
	 *
	 * @since 1.0
	 *
	 * @param string $title Title.
	 * @return string Modified title.
	 */
	function echoes_customize_page_title( $title ) {

		if ( is_home() && ( $blog_page_id = echoes_get_index_page_id( 'blog' ) ) > 0 ) {
			$title = get_the_title( $blog_page_id );
		}
		elseif ( is_singular() ) {
			$title = get_the_title();
		}
		elseif ( is_archive() ) {
			$title = strip_tags( get_the_archive_title() );
		}
		elseif ( is_search() ) {
			/* translators: %s: search query input */
			$title = sprintf( __( 'Search Results for: %s', 'echoes' ),  get_search_query() );
		}
		elseif ( is_404() ) {
			$title = __( '404!', 'echoes' );
		}
		return $title;
	}
endif;

add_filter( 'echoes_filter_custom_page_title', 'echoes_customize_page_title' );

if ( ! function_exists( 'echoes_add_author_bio_in_single' ) ) :

	/**
	 * Display Author bio.
	 *
	 * @since 1.0
	 */
	function echoes_add_author_bio_in_single() {

		if ( is_singular( 'post' ) ) {
			global $post;
			if ( get_the_author_meta( 'description', $post->post_author ) ) {
				get_template_part( 'template-parts/author-bio', 'single' );
			}
		}

	}

endif;

add_action( 'echoes_author_bio', 'echoes_add_author_bio_in_single' );

if ( ! function_exists( 'echoes_primary_navigation_fallback' ) ) :

	/**
	 * Fallback for primary navigation.
	 *
	 * @since 1.0
	 */
	function echoes_primary_navigation_fallback() {
		echo '<ul>';
		echo '<li><a href="' . esc_url( home_url( '/' ) ) . '">' . esc_html__( 'Home', 'echoes' ). '</a></li>';
		wp_list_pages( array(
			'title_li' => '',
			'depth'    => 1,
			'number'   => 7,
		) );
		echo '</ul>';

	}

endif;
