<?php
/**
 * Echoes functions and definitions.
 *
 * @package Echoes
 */

if ( ! function_exists( 'echoes_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function echoes_setup() {
		/*
		 * Make theme available for translation.
		 */
		load_theme_textdomain( 'echoes' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for custom logo.
		 */
		add_theme_support( 'custom-logo' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 */
		add_theme_support( 'post-thumbnails' );
		add_image_size( 'echoes-featured-banner', 1400, 320, true );

		// This theme uses wp_nav_menu() in four location.
		register_nav_menus( array(
			'primary'  => esc_html__( 'Primary Menu', 'echoes' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		// Set up the WordPress core custom background feature.
		add_theme_support( 'custom-background', apply_filters( 'echoes_custom_background_args', array(
			'default-color' => '#ffffff',
			'default-image' => '',
		) ) );

		// Set up the WordPress core custom header feature.
		add_theme_support( 'custom-header', apply_filters( 'echoes_custom_header_args', array(
				'default-image' => get_template_directory_uri() . '/assets/images/header-media.jpg',
				'width'         => 1400,
				'height'        => 320,
				'flex-height'   => true,
				'header-text'   => false,
		) ) );

		// Enable support for footer widgets.
		add_theme_support( 'footer-widgets', 4 );

		// Load Supports.
		require_once get_template_directory() . '/inc/footer-widget-area.php';

		global $echoes_default_options;
		$echoes_default_options = echoes_get_default_theme_options();

	}
endif;

add_action( 'after_setup_theme', 'echoes_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function echoes_content_width() {
	$GLOBALS['content_width'] = apply_filters( 'echoes_content_width', 640 );
}
add_action( 'after_setup_theme', 'echoes_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function echoes_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Primary Sidebar', 'echoes' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Add widgets here to appear in your Primary Sidebar.', 'echoes' ),
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="widget-title">',
		'after_title'   => '</h2>',
	) );

}
add_action( 'widgets_init', 'echoes_widgets_init' );

if ( ! function_exists( 'echoes_fonts_url' ) ) :

/**
* Register Google fonts.
*
* @since Echoes 1.0
*
* @return string Google fonts URL for the theme.
*/

function echoes_fonts_url() {

	$fonts_url = '';
	$fonts     = array();
	$subsets   = 'latin,latin-ext';
	
	// header fonts
	
	/* translators: If there are characters in your language that are not supported by Montserrat, translate this to 'off'. Do not translate into your own language. */
		
	if ( 'off' !== _x( 'on', 'Open Sans: on or off', 'echoes' ) ) {
		$fonts[] = 'Open Sans:300,400,500,600,700';
	}

	if ( $fonts ) {
		$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $fonts ) ),
			'subset' => urlencode( $subsets ),
		), '//fonts.googleapis.com/css' );
	}

	return $fonts_url;

}

endif;

/**
 * Enqueue scripts and styles.
 */
function echoes_scripts() {

	$fonts_url = echoes_fonts_url();
	if ( ! empty( $fonts_url ) ) {
		wp_enqueue_style( 'echoes-google-fonts', $fonts_url, array(), null );
	}

	wp_enqueue_style( 'echoes-style', get_stylesheet_uri(), array(), '1.4.0' );
        
	wp_enqueue_script( 'echoes-skip-link-focus-fix', get_template_directory_uri() . '/assets/js/skip-link-focus-fix.js', array(), '1.2.0', true );

	wp_enqueue_script( 'echoes-custom', get_template_directory_uri() . '/assets/js/custom.js', array( 'jquery' ), '1.2.0', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
        
        /** Include conditional css **/
        require_once get_template_directory() . '/condition_css.php';
}
add_action( 'wp_enqueue_scripts', 'echoes_scripts' );

/**
 * Include default theme options.
 */
require_once get_template_directory() . '/inc/customizer/default.php';

/**
 * Load Layout Option.
 */
require_once get_template_directory() . '/inc/layout-option.php';

/**
 * Load theme core functions.
 */
require_once get_template_directory() . '/inc/core.php';

/**
 * Load hooks.
 */
require_once get_template_directory() . '/inc/functions/structure.php';
require_once get_template_directory() . '/inc/functions/basic.php';
require_once get_template_directory() . '/inc/functions/custom.php';

/**
 * Custom template tags for this theme.
 */
require_once get_template_directory() . '/inc/template-tags.php';

/**
 * Customizer additions.
 */
require_once get_template_directory() . '/inc/customizer.php';

//Add Menu Parent Class
function echoes_add_menu_parent_class( $items ) {
    $parents = array();

    foreach ( $items as $item ) {
        //Check if the item is a parent item
        if ( $item->menu_item_parent && $item->menu_item_parent > 0 ) {
          $parents[] = $item->menu_item_parent;
        }
    }

    foreach ( $items as $item ) {
        if ( in_array( $item->ID, $parents ) ) {
          //Add "dropdown" class to parents
          $item->classes[] = 'dropdown';
        }
    }

    return $items;
}

/**
 * Excerpt.
 */

if ( ! function_exists( 'echoes_excerpt_length' ) ) :
	/**
	 * excerpt length.
	 *
	 * @param int $length in words.
	 *
	 */
	function echoes_excerpt_length( $length ) {
		if ( is_admin() ) {
			return $length;
		}

		$length = echoes_get_option( 'excerpt_length' );
		$length = ! empty( $length ) ? $length : '25';
		return $length;
	}
endif;
add_filter( 'excerpt_length', 'echoes_excerpt_length', 999 );

if( ! function_exists( 'echoes_excerpt_more' ) ) :
	/**
	 * @param string $more.
	 *
	 * Return string 
	 */
	function echoes_excerpt_more( $more ) {
		if ( is_admin() ) {
			return $more;
		}
		/* translators: %s: post title */
		$more =	sprintf( __( '<span class="screen-reader-text"> "%s"</span>', 'echoes' ), get_the_title( get_the_ID() ) );
		return ' &hellip; ' . $more;
	}
endif;	
add_filter('excerpt_more', 'echoes_excerpt_more');

/**
* Get sidebar options.
*/

if ( ! function_exists( 'echoes_get_sidebar_options' ) ) :

	function echoes_get_sidebar_options() {

		global $wp_registered_sidebars;

		$output = array();

		if ( ! empty( $wp_registered_sidebars ) && is_array( $wp_registered_sidebars ) ) {
			foreach ( $wp_registered_sidebars as $key => $sidebar ) {
				$output[ $key ] = $sidebar['name'];
			}
		}

		return $output;

	}

endif;

if ( ! function_exists( 'echoes_get_index_page_id' ) ) :

	/**
	 * Get front index page ID.
	 *
	 * @param string $type Type.
	 * @return int Corresponding Page ID.
	 */
	function echoes_get_index_page_id( $type = 'front' ) {

		$page = '';

		switch ( $type ) {
			case 'front':
				$page = get_option( 'page_on_front' );
				break;

			case 'blog':
				$page = get_option( 'page_for_posts' );
				break;

			default:
				break;
		}
		$page = absint( $page );
		return $page;

	}
endif;

if ( ! function_exists( 'echoes_the_custom_logo' ) ) :

	/**
	 * Render logo.
	 */
	function echoes_the_custom_logo() {

		if ( function_exists( 'the_custom_logo' ) ) {
			the_custom_logo();
		}

	}

endif;