/**
 * ChronoCards Ruchir Theme JavaScript
 * 
 * Handles timeline card interactions and animations
 */

(function($) {
    'use strict';

    // Initialize when document is ready
    $(document).ready(function() {
        chronocards_ruchir_initTimelineCards();
        chronocards_ruchir_initMobileMenu();
        chronocards_ruchir_initDesktopMenu();
        chronocards_ruchir_initSmoothScrolling();
        chronocards_ruchir_initLazyLoading();
        chronocards_ruchir_initSkipLink();
        chronocards_ruchir_initMediaKeyboardNavigation();
        
        // Ensure sub-menus are hidden on mobile by default
        if ($(window).width() <= 768) {
            $('.main-navigation li').removeClass('submenu-open');
        }
    });

    /**
     * Initialize timeline card functionality
     */
    function chronocards_ruchir_initTimelineCards() {
        $('.timeline-card').each(function() {
            var $card = $(this);
            var $content = $card.find('.card-content');
            var $fullContent = $card.find('.card-full-content');
            
            // Add click handler for card expansion (but allow links to work normally)
            $card.on('click', function(e) {
                var $target = $(e.target);
                
                // Don't prevent default if clicking on a link or image
                if ($target.is('a') || $target.closest('a').length > 0 || $target.is('img') || $target.closest('img').length > 0) {
                    return; // Allow normal navigation
                }
                
                e.preventDefault();
                
                if ($card.hasClass('expanded')) {
                    chronocards_ruchir_collapseCard($card);
                } else {
                    chronocards_ruchir_expandCard($card);
                }
            });

            // Add hover effects
            $card.hover(
                function() {
                    $(this).addClass('hovered');
                },
                function() {
                    $(this).removeClass('hovered');
                }
            );
        });
    }

    /**
     * Enhance Skip to content link - ensure main content is focusable and focused
     * Note: Skip link works natively via href="#primary", this ensures proper focus
     */
    function chronocards_ruchir_initSkipLink() {
        // Ensure main content element is focusable for skip link
        var $main = $('#primary');
        if ($main.length) {
            // Always set tabindex to make it focusable
            $main.attr('tabindex', '-1');
            
            // Handle skip link click and keyboard activation (Enter/Space)
            $('.skip-link').on('click', function(e) {
                var $link = $(this);
                var targetId = $link.attr('href');
                
                if (targetId && targetId.indexOf('#') === 0) {
                    var $target = $(targetId);
                    if ($target.length) {
                        // Prevent default to handle focus manually
                        e.preventDefault();
                        
                        // Ensure target is focusable
                        if (!$target.is('[tabindex]')) {
                            $target.attr('tabindex', '-1');
                        }
                        
                        // Update URL hash for browser history
                        if (history.pushState) {
                            history.pushState(null, null, targetId);
                        } else {
                            window.location.hash = targetId;
                        }
                        
                        // Scroll to target first
                        $target[0].scrollIntoView({ behavior: 'smooth', block: 'start' });
                        
                        // Focus after scroll completes - use multiple attempts to ensure it works
                        setTimeout(function() {
                            $target[0].focus();
                            // Force browser to apply focus styles
                            $target[0].blur();
                            $target[0].focus();
                        }, 200);
                    }
                }
            });
            
            // Also handle keyboard activation (Enter/Space on skip link)
            $('.skip-link').on('keydown', function(e) {
                if (e.key === 'Enter' || e.key === ' ') {
                    var $link = $(this);
                    var targetId = $link.attr('href');
                    
                    if (targetId && targetId.indexOf('#') === 0) {
                        var $target = $(targetId);
                        if ($target.length) {
                            e.preventDefault();
                            
                            // Ensure target is focusable
                            if (!$target.is('[tabindex]')) {
                                $target.attr('tabindex', '-1');
                            }
                            
                            // Update URL hash
                            if (history.pushState) {
                                history.pushState(null, null, targetId);
                            } else {
                                window.location.hash = targetId;
                            }
                            
                            // Scroll and focus
                            $target[0].scrollIntoView({ behavior: 'smooth', block: 'start' });
                            setTimeout(function() {
                                $target[0].focus();
                                $target[0].blur();
                                $target[0].focus();
                            }, 200);
                        }
                    }
                }
            });
            
            // Handle hash change in case user navigates directly
            $(window).on('hashchange', function() {
                if (window.location.hash === '#primary') {
                    setTimeout(function() {
                        if ($main.length) {
                            if (!$main.is('[tabindex]')) {
                                $main.attr('tabindex', '-1');
                            }
                            $main[0].focus();
                            $main[0].scrollIntoView({ behavior: 'smooth', block: 'start' });
                        }
                    }, 100);
                }
            });
            
            // Handle initial page load with hash
            if (window.location.hash === '#primary') {
                setTimeout(function() {
                    if ($main.length) {
                        if (!$main.is('[tabindex]')) {
                            $main.attr('tabindex', '-1');
                        }
                        $main[0].focus();
                    }
                }, 500);
            }
        }
    }

    /**
     * Expand a timeline card
     */
    function chronocards_ruchir_expandCard($card) {
        var postId = $card.attr('id').replace('post-', '');
        
        // Add loading state
        $card.addClass('loading');
        
        // AJAX request to get full content
        $.ajax({
            url: chronocards_ajax.ajax_url,
            type: 'POST',
            data: {
                action: 'expand_card',
                post_id: postId,
                nonce: chronocards_ajax.nonce
            },
            success: function(response) {
                if (response.success) {
                    var $fullContent = $card.find('.card-full-content');
                    $fullContent.html(response.data.content);
                    $card.addClass('expanded');
                    
                    // Re-initialize media keyboard navigation for dynamically loaded content
                    chronocards_ruchir_initMediaKeyboardNavigation();
                    
                    // Animate expansion
                    chronocards_ruchir_animateCardExpansion($card);
                }
            },
            error: function() {
                // Error loading card content
            },
            complete: function() {
                $card.removeClass('loading');
            }
        });
    }

    /**
     * Collapse a timeline card
     */
    function chronocards_ruchir_collapseCard($card) {
        $card.removeClass('expanded');
        
        // Animate collapse
        chronocards_ruchir_animateCardCollapse($card);
    }

    /**
     * Animate card expansion
     */
    function chronocards_ruchir_animateCardExpansion($card) {
        $card.css({
            'transform': 'scale(1.02)',
            'transition': 'transform 0.3s ease'
        });
        
        setTimeout(function() {
            $card.css('transform', 'scale(1)');
        }, 300);
    }

    /**
     * Animate card collapse
     */
    function chronocards_ruchir_animateCardCollapse($card) {
        $card.css({
            'transform': 'scale(0.98)',
            'transition': 'transform 0.3s ease'
        });
        
        setTimeout(function() {
            $card.css('transform', 'scale(1)');
        }, 300);
    }

        /**
         * Initialize mobile menu toggle
         */
        function chronocards_ruchir_initMobileMenu() {
            $('.menu-toggle').on('click', function(e) {
                e.preventDefault();
                var $menu = $('.main-navigation');
                var $button = $(this);
                var $closeButton = $('.menu-close');
                
                // Toggle menu
                $menu.toggleClass('toggled');
                $button.toggleClass('toggled');
                
                var expanded = $menu.hasClass('toggled');
                $button.attr('aria-expanded', expanded);
                
                // Show/hide close button
                if (expanded) {
                    $closeButton.show();
                    $('body').css('overflow', 'hidden');
                    chronocards_ruchir_trapFocusInMobileMenu();
                    // Move focus to first link inside the menu for accessibility
                    var $firstFocusable = chronocards_ruchir_getMobileMenuFocusables().first();
                    if ($firstFocusable.length) {
                        setTimeout(function(){ $firstFocusable.focus(); }, 50);
                    }
                } else {
                    $closeButton.hide();
                    $('body').css('overflow', '');
                }
            });

            // Handle close button click
            $('.menu-close').on('click', function() {
                $('.main-navigation').removeClass('toggled');
                $('.menu-toggle').removeClass('toggled').attr('aria-expanded', 'false');
                $('.menu-close').hide();
                $('body').css('overflow', '');
            });

            // Handle mobile sub-menu toggling (click and keyboard)
            $(document).on('click keydown', '.main-navigation a', function(e) {
                if ($(window).width() <= 768) {
                    var $parent = $(this).parent();
                    var $submenu = $parent.find('> ul, > .sub-menu');
                    
                    // Handle both click and Enter/Space key
                    if ($submenu.length > 0 && (e.type === 'click' || e.key === 'Enter' || e.key === ' ')) {
                        e.preventDefault();
                        $parent.toggleClass('submenu-open');
                        
                        // Update ARIA attributes
                        var $link = $(this);
                        var isExpanded = $parent.hasClass('submenu-open');
                        $link.attr('aria-expanded', isExpanded);
                        
                        // Focus management for keyboard navigation
                        if (e.type === 'keydown') {
                            var $firstSubLink = $submenu.find('a').first();
                            if (isExpanded && $firstSubLink.length) {
                                setTimeout(function() {
                                    $firstSubLink.focus();
                                }, 100);
                            }
                        }
                    }
                }
            });

            // Close menu when clicking outside
            $(document).on('click', function(e) {
                if (!$(e.target).closest('.main-navigation, .menu-toggle, .menu-close').length) {
                    $('.main-navigation').removeClass('toggled');
                    $('.menu-toggle').removeClass('toggled').attr('aria-expanded', 'false');
                    $('.menu-close').hide();
                    $('body').css('overflow', '');
                }
            });

            // Handle escape key
            $(document).on('keydown', function(e) {
                if (e.key === 'Escape' && $('.main-navigation').hasClass('toggled')) {
                    $('.main-navigation').removeClass('toggled');
                    $('.menu-toggle').removeClass('toggled').attr('aria-expanded', 'false').focus();
                    $('.menu-close').hide();
                    $('body').css('overflow', '');
                }
            });
        }

        /**
         * Trap focus within the mobile menu when open
         */
        function chronocards_ruchir_trapFocusInMobileMenu() {
            var $menu = $('.main-navigation');
            var $closeButton = $('.menu-close');
            var lastShiftTabFromClose = false;

            // Cleanup previous handlers
            $(document).off('keydown.mobileFocusTrap');

            $(document).on('keydown.mobileFocusTrap', function(e) {
                if (!$menu.hasClass('toggled')) {
                    return;
                }

                if (e.key !== 'Tab') {
                    return;
                }

                var $focusables = chronocards_ruchir_getMobileMenuFocusables();
                if ($focusables.length === 0) {
                    return;
                }

                var $first = $focusables.first();
                var $last = $focusables.last();
                var isShift = e.shiftKey;
                var $active = $(document.activeElement);

                // If focus is on close button and Shift+Tab, go to last focusable in menu
                if ($active.is($closeButton) && isShift) {
                    e.preventDefault();
                    $last.focus();
                    lastShiftTabFromClose = true;
                    return;
                }

                // If focus moved to body or outside, send it to first
                if (!$active.length || !$active.closest('.main-navigation, .menu-close').length) {
                    e.preventDefault();
                    $first.focus();
                    return;
                }

                // Tabbing forward from last element goes to close button
                if (!isShift && $active.is($last)) {
                    e.preventDefault();
                    $closeButton.focus();
                    return;
                }

                // Shift+Tab from first element goes to close button
                if (isShift && $active.is($first)) {
                    e.preventDefault();
                    $closeButton.focus();
                    return;
                }
            });

            // When menu closes, remove trap
            $(document).one('click keydown', function() {
                if (!$('.main-navigation').hasClass('toggled')) {
                    $(document).off('keydown.mobileFocusTrap');
                }
            });
        }

        /**
         * Get focusable elements inside mobile menu
         */
        function chronocards_ruchir_getMobileMenuFocusables() {
            var selectors = 'a[href], button:not([disabled]), [tabindex]:not([tabindex="-1"])';
            var $inMenu = $('.main-navigation.toggled').find(selectors).filter(':visible');
            // Exclude the toggle button if still visible and include menu-close handled separately
            $inMenu = $inMenu.not('.menu-toggle');
            return $inMenu;
        }

    /**
     * Initialize desktop menu interactions
     */
    function chronocards_ruchir_initDesktopMenu() {
        // Simple hover functionality - let CSS handle the rest
        $('.main-navigation li.menu-item-has-children').hover(
            function() {
                $(this).addClass('hovered');
            },
            function() {
                $(this).removeClass('hovered');
            }
        );

        // Keyboard navigation for desktop submenus
        $('.main-navigation li.menu-item-has-children > a').on('keydown', function(e) {
            var $parent = $(this).parent();
            var $submenu = $parent.find('> ul, > .sub-menu');
            
            if ($submenu.length > 0) {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    // Toggle submenu visibility
                    $parent.toggleClass('submenu-open');
                    var isExpanded = $parent.hasClass('submenu-open');
                    $(this).attr('aria-expanded', isExpanded);
                    
                    if (isExpanded) {
                        // Focus first submenu item
                        var $firstSubLink = $submenu.find('a').first();
                        if ($firstSubLink.length) {
                            setTimeout(function() {
                                $firstSubLink.focus();
                            }, 100);
                        }
                    }
                } else if (e.key === 'ArrowDown') {
                    e.preventDefault();
                    // Open submenu and focus first item
                    $parent.addClass('submenu-open');
                    $(this).attr('aria-expanded', 'true');
                    var $firstSubLink = $submenu.find('a').first();
                    if ($firstSubLink.length) {
                        setTimeout(function() {
                            $firstSubLink.focus();
                        }, 100);
                    }
                }
            }
        });

        // Keyboard navigation within submenus
        $('.main-navigation ul ul a, .main-navigation .sub-menu a').on('keydown', function(e) {
            var $current = $(this);
            var $submenu = $current.closest('ul, .sub-menu');
            var $items = $submenu.find('a');
            var currentIndex = $items.index($current);

            if (e.key === 'ArrowDown') {
                e.preventDefault();
                var nextIndex = (currentIndex + 1) % $items.length;
                $items.eq(nextIndex).focus();
            } else if (e.key === 'ArrowUp') {
                e.preventDefault();
                var prevIndex = currentIndex > 0 ? currentIndex - 1 : $items.length - 1;
                $items.eq(prevIndex).focus();
            } else if (e.key === 'Escape') {
                e.preventDefault();
                // Close submenu and return focus to parent
                var $parent = $current.closest('li.menu-item-has-children');
                $parent.removeClass('submenu-open');
                $parent.find('> a').attr('aria-expanded', 'false').focus();
            } else if (e.key === 'ArrowLeft' || e.key === 'ArrowRight') {
                e.preventDefault();
                // Close submenu and move to sibling menu items
                var $parent = $current.closest('li.menu-item-has-children');
                $parent.removeClass('submenu-open');
                $parent.find('> a').attr('aria-expanded', 'false');
                
                var $allTopLevelItems = $('.main-navigation > ul > li > a, .main-navigation .nav-menu > li > a');
                var parentIndex = $allTopLevelItems.index($parent.find('> a'));
                
                if (e.key === 'ArrowLeft') {
                    var prevTopIndex = parentIndex > 0 ? parentIndex - 1 : $allTopLevelItems.length - 1;
                    $allTopLevelItems.eq(prevTopIndex).focus();
                } else {
                    var nextTopIndex = (parentIndex + 1) % $allTopLevelItems.length;
                    $allTopLevelItems.eq(nextTopIndex).focus();
                }
            }
        });
    }

    /**
     * Initialize smooth scrolling for anchor links
     */
    function chronocards_ruchir_initSmoothScrolling() {
        $('a[href*="#"]:not([href="#"])').click(function() {
            if (location.pathname.replace(/^\//, '') == this.pathname.replace(/^\//, '') && location.hostname == this.hostname) {
                var target = $(this.hash);
                target = target.length ? target : $('[name=' + this.hash.slice(1) + ']');
                if (target.length) {
                    $('html, body').animate({
                        scrollTop: target.offset().top - 100
                    }, 1000);
                    return false;
                }
            }
        });
    }

    /**
     * Initialize lazy loading for images
     */
    function chronocards_ruchir_initLazyLoading() {
        if ('IntersectionObserver' in window) {
            var imageObserver = new IntersectionObserver(function(entries, observer) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        var img = entry.target;
                        img.src = img.dataset.src;
                        img.classList.remove('lazy');
                        imageObserver.unobserve(img);
                    }
                });
            });

            $('.lazy').each(function() {
                imageObserver.observe(this);
            });
        }
    }

    /**
     * Initialize timeline animation on scroll
     */
    function chronocards_ruchir_initTimelineAnimation() {
        if ('IntersectionObserver' in window) {
            var timelineObserver = new IntersectionObserver(function(entries) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('animate-in');
                    }
                });
            }, {
                threshold: 0.1
            });

            $('.timeline-card').each(function() {
                timelineObserver.observe(this);
            });
        }
    }

    /**
     * Initialize category color detection and featured image color extraction
     */
    function chronocards_ruchir_initCategoryColors() {
        $('.timeline-card').each(function() {
            var $card = $(this);
            var $image = $card.find('.card-image');
            
            if ($image.length) {
                // Extract dominant color from featured image
                chronocards_ruchir_extractDominantColor($image.attr('src'), function(color) {
                    if (color) {
                        $card.find('.timeline-dot').css('background-color', color);
                        // Also apply color to card background with low opacity
                        $card.css('background-color', chronocards_ruchir_hexToRgba(color, 0.1));
                    }
                });
            }
        });
    }

    /**
     * Extract dominant color from image
     */
    function chronocards_ruchir_extractDominantColor(imageSrc, callback) {
        var img = new Image();
        img.crossOrigin = 'anonymous';
        
        img.onload = function() {
            try {
                var canvas = document.createElement('canvas');
                var ctx = canvas.getContext('2d');
                
                // Resize image for faster processing
                var maxSize = 50;
                var ratio = Math.min(maxSize / img.width, maxSize / img.height);
                canvas.width = img.width * ratio;
                canvas.height = img.height * ratio;
                
                ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
                
                var imageData = ctx.getImageData(0, 0, canvas.width, canvas.height);
                var data = imageData.data;
                
                // Calculate average color
                var r = 0, g = 0, b = 0;
                var pixelCount = 0;
                
                for (var i = 0; i < data.length; i += 4) {
                    r += data[i];
                    g += data[i + 1];
                    b += data[i + 2];
                    pixelCount++;
                }
                
                r = Math.floor(r / pixelCount);
                g = Math.floor(g / pixelCount);
                b = Math.floor(b / pixelCount);
                
                var hexColor = '#' + 
                    ('0' + r.toString(16)).slice(-2) +
                    ('0' + g.toString(16)).slice(-2) +
                    ('0' + b.toString(16)).slice(-2);
                
                callback(hexColor);
            } catch (e) {
                callback(null);
            }
        };
        
        img.onerror = function() {
            callback(null);
        };
        
        img.src = imageSrc;
    }

    /**
     * Convert hex to rgba
     */
    function chronocards_ruchir_hexToRgba(hex, alpha) {
        var r = parseInt(hex.slice(1, 3), 16);
        var g = parseInt(hex.slice(3, 5), 16);
        var b = parseInt(hex.slice(5, 7), 16);
        return 'rgba(' + r + ', ' + g + ', ' + b + ', ' + alpha + ')';
    }

    /**
     * Initialize keyboard navigation
     */
    function chronocards_ruchir_initKeyboardNavigation() {
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape') {
                // Close expanded cards
                $('.timeline-card.expanded').removeClass('expanded');
                
                // Close mobile menu
                $('.main-navigation').removeClass('toggled');
                $('.menu-toggle').removeClass('toggled').attr('aria-expanded', 'false');
            }
        });
    }

    /**
     * Initialize accessibility features
     */
    function chronocards_ruchir_initAccessibility() {
        // Add ARIA labels to interactive elements
        $('.timeline-card').attr('role', 'button');
        $('.timeline-card').attr('tabindex', '0');
        
        // Add keyboard support for cards
        $('.timeline-card').on('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                $(this).click();
            }
        });

        // Ensure images and media elements are keyboard accessible on mobile
        chronocards_ruchir_initMediaKeyboardNavigation();
    }

    /**
     * Initialize keyboard navigation for media elements (images, videos, etc.)
     * Ensures media elements are keyboard accessible, especially on mobile
     */
    function chronocards_ruchir_initMediaKeyboardNavigation() {
        // Selector for all content areas that may contain media
        var contentSelectors = '.entry-content, .comment-content, .widget_text, .card-content, .card-full-content';
        
        // Make images keyboard accessible
        $(contentSelectors + ' img').each(function() {
            var $img = $(this);
            var $parent = $img.parent();
            
            // If image is not inside a link, make it keyboard accessible
            if (!$parent.is('a') && !$img.closest('a').length) {
                // Add tabindex for keyboard navigation (especially important on mobile)
                if (!$img.attr('tabindex')) {
                    $img.attr('tabindex', '0');
                }
                $img.attr('role', 'img');
                
                // Ensure image has alt text for accessibility
                if (!$img.attr('alt')) {
                    $img.attr('alt', '');
                }
                
                // Add keyboard support for images
                $img.off('keydown.mediaNav').on('keydown.mediaNav', function(e) {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        // Focus the image (for screen readers and mobile keyboard navigation)
                        $img.focus();
                        // On mobile, ensure image is visible when focused
                        if ($(window).width() <= 768) {
                            $img[0].scrollIntoView({ behavior: 'smooth', block: 'center' });
                        }
                    }
                });
            } else {
                // If image is in a link, ensure the link is keyboard accessible
                var $link = $img.closest('a');
                if ($link.length && !$link.attr('tabindex')) {
                    $link.attr('tabindex', '0');
                }
            }
        });

        // Handle card images specifically (featured images in timeline cards)
        $('.card-image').each(function() {
            var $img = $(this);
            var $link = $img.closest('a');
            
            // Ensure the link wrapping card images is keyboard accessible
            if ($link.length) {
                if (!$link.attr('tabindex')) {
                    $link.attr('tabindex', '0');
                }
                // Add keyboard support for card image links on mobile
                if ($(window).width() <= 768) {
                    $link.off('keydown.cardImage').on('keydown.cardImage', function(e) {
                        if (e.key === 'Enter' || e.key === ' ') {
                            // Allow default navigation, but ensure focus is visible
                            setTimeout(function() {
                                $link.focus();
                            }, 100);
                        }
                    });
                }
            }
        });

        // Ensure videos are keyboard accessible with proper controls
        $(contentSelectors + ' video').each(function() {
            var $video = $(this);
            if (!$video.attr('tabindex')) {
                $video.attr('tabindex', '0');
            }
            // Ensure video has controls for keyboard users
            if (!$video.attr('controls')) {
                $video.attr('controls', 'controls');
            }
            // Add keyboard support for video controls on mobile
            if ($(window).width() <= 768) {
                $video.off('keydown.videoNav').on('keydown.videoNav', function(e) {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        if (this.paused) {
                            this.play();
                        } else {
                            this.pause();
                        }
                        $video.focus();
                    }
                });
            }
        });

        // Ensure iframes and embeds are keyboard accessible
        $(contentSelectors + ' iframe, ' + contentSelectors + ' embed, ' + contentSelectors + ' object').each(function() {
            var $media = $(this);
            if (!$media.attr('tabindex')) {
                $media.attr('tabindex', '0');
            }
            // Add title for accessibility if missing
            if (!$media.attr('title')) {
                $media.attr('title', 'Embedded content');
            }
        });

        // Ensure audio elements are keyboard accessible with proper controls
        $(contentSelectors + ' audio').each(function() {
            var $audio = $(this);
            if (!$audio.attr('tabindex')) {
                $audio.attr('tabindex', '0');
            }
            // Ensure audio has controls for keyboard users
            if (!$audio.attr('controls')) {
                $audio.attr('controls', 'controls');
            }
            // Add keyboard support for audio controls on mobile
            if ($(window).width() <= 768) {
                $audio.off('keydown.audioNav').on('keydown.audioNav', function(e) {
                    if (e.key === 'Enter' || e.key === ' ') {
                        e.preventDefault();
                        if (this.paused) {
                            this.play();
                        } else {
                            this.pause();
                        }
                        $audio.focus();
                    }
                });
            }
        });
    }

    // Initialize additional features
    $(window).on('load', function() {
        chronocards_ruchir_initTimelineAnimation();
        chronocards_ruchir_initCategoryColors();
        chronocards_ruchir_initKeyboardNavigation();
        chronocards_ruchir_initAccessibility();
        chronocards_ruchir_initScrollEffects();
        chronocards_ruchir_initCardInteractions();
    });

    /**
     * Initialize scroll effects for timeline
     */
    function chronocards_ruchir_initScrollEffects() {
        $(window).on('scroll', function() {
            var scrollTop = $(window).scrollTop();
            var windowHeight = $(window).height();
            
            $('.timeline-card').each(function() {
                var $card = $(this);
                var cardTop = $card.offset().top;
                var cardHeight = $card.outerHeight();
                
                // Add parallax effect to timeline line
                var parallaxOffset = (scrollTop - cardTop) * 0.1;
                $('.timeline-line').css('transform', 'translateX(-50%) translateY(' + parallaxOffset + 'px)');
                
                // Animate cards on scroll
                if (cardTop < scrollTop + windowHeight - 100) {
                    $card.addClass('animate-in');
                }
            });
        });
    }

    /**
     * Initialize enhanced card interactions
     */
    function chronocards_ruchir_initCardInteractions() {
        // Add click-to-expand functionality (but allow links to work normally)
        $('.timeline-card').on('click', function(e) {
            var $card = $(this);
            var $target = $(e.target);
            
            // Don't prevent default if clicking on a link or image
            if ($target.is('a') || $target.closest('a').length > 0 || $target.is('img') || $target.closest('img').length > 0) {
                return; // Allow normal navigation
            }
            
            e.preventDefault();
            
            if ($card.hasClass('expanded')) {
                chronocards_ruchir_collapseCard($card);
            } else {
                // Collapse other expanded cards first
                $('.timeline-card.expanded').not($card).removeClass('expanded');
                chronocards_ruchir_expandCard($card);
            }
        });

        // Add keyboard support for card expansion
        $('.timeline-card').on('keydown', function(e) {
            if (e.key === 'Enter' || e.key === ' ') {
                e.preventDefault();
                $(this).click();
            }
        });

        // Add hover effects for better UX
        $('.timeline-card').hover(
            function() {
                $(this).addClass('hovered');
            },
            function() {
                $(this).removeClass('hovered');
            }
        );
    }

    // Handle window resize
    $(window).on('resize', function() {
        // Recalculate timeline positions if needed
        $('.timeline-card').removeClass('animate-in');
        setTimeout(chronocards_ruchir_initTimelineAnimation, 100);
    });

})(jQuery);
