<?php
/**
 * ChronoCards Ruchir Theme Customizer
 *
 * @package ChronoCards_Ruchir
 */

/**
 * Add postMessage support for site title and description for the Theme Customizer.
 *
 * @param WP_Customize_Manager $wp_customize Theme Customizer object.
 */
function chronocards_ruchir_customize_register( $wp_customize ) {
    $wp_customize->get_setting( 'blogname' )->transport         = 'postMessage';
    $wp_customize->get_setting( 'blogdescription' )->transport  = 'postMessage';

    if ( isset( $wp_customize->selective_refresh ) ) {
        $wp_customize->selective_refresh->add_partial( 'blogname', array(
            'selector'        => '.site-title a',
            'render_callback' => 'chronocards_ruchir_customize_partial_blogname',
        ) );
        $wp_customize->selective_refresh->add_partial( 'blogdescription', array(
            'selector'        => '.site-description',
            'render_callback' => 'chronocards_ruchir_customize_partial_blogdescription',
        ) );
    }

    // Add Timeline Settings section
    $wp_customize->add_section( 'chronocards_timeline', array(
        'title'    => __( 'Timeline Settings', 'chronocards-ruchir' ),
        'priority' => 30,
    ) );

    // Timeline line color
    $wp_customize->add_setting( 'timeline_line_color', array(
        'default'           => '#3498db',
        'sanitize_callback' => 'sanitize_hex_color',
        'transport'         => 'postMessage',
    ) );

    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'timeline_line_color', array(
        'label'    => __( 'Timeline Line Color', 'chronocards-ruchir' ),
        'section'  => 'chronocards_timeline',
        'settings' => 'timeline_line_color',
    ) ) );

    // Card background color
    $wp_customize->add_setting( 'card_background_color', array(
        'default'           => '#ffffff',
        'sanitize_callback' => 'sanitize_hex_color',
        'transport'         => 'postMessage',
    ) );

    $wp_customize->add_control( new WP_Customize_Color_Control( $wp_customize, 'card_background_color', array(
        'label'    => __( 'Card Background Color', 'chronocards-ruchir' ),
        'section'  => 'chronocards_timeline',
        'settings' => 'card_background_color',
    ) ) );

    // Card border radius
    $wp_customize->add_setting( 'card_border_radius', array(
        'default'           => '15',
        'sanitize_callback' => 'absint',
        'transport'         => 'postMessage',
    ) );

    $wp_customize->add_control( 'card_border_radius', array(
        'label'       => __( 'Card Border Radius (px)', 'chronocards-ruchir' ),
        'section'     => 'chronocards_timeline',
        'type'        => 'number',
        'input_attrs' => array(
            'min'  => 0,
            'max'  => 50,
            'step' => 1,
        ),
    ) );

    // Timeline dot size
    $wp_customize->add_setting( 'timeline_dot_size', array(
        'default'           => '20',
        'sanitize_callback' => 'absint',
        'transport'         => 'postMessage',
    ) );

    $wp_customize->add_control( 'timeline_dot_size', array(
        'label'       => __( 'Timeline Dot Size (px)', 'chronocards-ruchir' ),
        'section'     => 'chronocards_timeline',
        'type'        => 'number',
        'input_attrs' => array(
            'min'  => 10,
            'max'  => 40,
            'step' => 1,
        ),
    ) );

    // Add Typography section
    $wp_customize->add_section( 'chronocards_typography', array(
        'title'    => __( 'Typography', 'chronocards-ruchir' ),
        'priority' => 35,
    ) );

    // Site title font size
    $wp_customize->add_setting( 'site_title_font_size', array(
        'default'           => '2.5',
        'sanitize_callback' => 'chronocards_ruchir_sanitize_font_size',
        'transport'         => 'postMessage',
    ) );

    $wp_customize->add_control( 'site_title_font_size', array(
        'label'       => __( 'Site Title Font Size (rem)', 'chronocards-ruchir' ),
        'section'     => 'chronocards_typography',
        'type'        => 'text',
    ) );

    // Card title font size
    $wp_customize->add_setting( 'card_title_font_size', array(
        'default'           => '1.5',
        'sanitize_callback' => 'chronocards_ruchir_sanitize_font_size',
        'transport'         => 'postMessage',
    ) );

    $wp_customize->add_control( 'card_title_font_size', array(
        'label'       => __( 'Card Title Font Size (rem)', 'chronocards-ruchir' ),
        'section'     => 'chronocards_typography',
        'type'        => 'text',
    ) );

    // Add Layout section
    $wp_customize->add_section( 'chronocards_layout', array(
        'title'    => __( 'Layout Settings', 'chronocards-ruchir' ),
        'priority' => 40,
    ) );

    // Container max width
    $wp_customize->add_setting( 'container_max_width', array(
        'default'           => '1200',
        'sanitize_callback' => 'absint',
        'transport'         => 'postMessage',
    ) );

    $wp_customize->add_control( 'container_max_width', array(
        'label'       => __( 'Container Max Width (px)', 'chronocards-ruchir' ),
        'section'     => 'chronocards_layout',
        'type'        => 'number',
        'input_attrs' => array(
            'min'  => 800,
            'max'  => 1600,
            'step' => 50,
        ),
    ) );

    // Card spacing
    $wp_customize->add_setting( 'card_spacing', array(
        'default'           => '40',
        'sanitize_callback' => 'absint',
        'transport'         => 'postMessage',
    ) );

    $wp_customize->add_control( 'card_spacing', array(
        'label'       => __( 'Card Spacing (px)', 'chronocards-ruchir' ),
        'section'     => 'chronocards_layout',
        'type'        => 'number',
        'input_attrs' => array(
            'min'  => 20,
            'max'  => 100,
            'step' => 5,
        ),
    ) );
}
add_action( 'customize_register', 'chronocards_ruchir_customize_register' );

/**
 * Sanitize font size value
 * Validates that the value is a positive number (can be decimal for rem units)
 *
 * @param string $value Font size value to sanitize.
 * @return string Sanitized font size value.
 */
function chronocards_ruchir_sanitize_font_size( $value ) {
    // Remove any whitespace
    $value = trim( $value );
    
    // Convert to string if not already
    $value = (string) $value;
    
    // Check if value is numeric (including decimals)
    if ( is_numeric( $value ) ) {
        $float_value = floatval( $value );
        // Ensure it's positive and within reasonable range (0.1 to 10 rem)
        if ( $float_value > 0 && $float_value <= 10 ) {
            return $value;
        }
    }
    
    // Return default if invalid
    return '1.5';
}

/**
 * Render the site title for the selective refresh partial.
 *
 * @return void
 */
function chronocards_ruchir_customize_partial_blogname() {
    bloginfo( 'name' );
}

/**
 * Render the site tagline for the selective refresh partial.
 *
 * @return void
 */
function chronocards_ruchir_customize_partial_blogdescription() {
    bloginfo( 'description' );
}

/**
 * Binds JS handlers to make Theme Customizer preview reload changes asynchronously.
 */
function chronocards_ruchir_customize_preview_js() {
    wp_enqueue_script( 'chronocards-ruchir-customizer', get_template_directory_uri() . '/js/customizer.js', array( 'customize-preview' ), wp_get_theme()->get( 'Version' ), true );
}
add_action( 'customize_preview_init', 'chronocards_ruchir_customize_preview_js' );

/**
 * Add custom CSS from customizer settings
 */
function chronocards_ruchir_customizer_css() {
    $timeline_color = sanitize_hex_color( get_theme_mod( 'timeline_line_color', '#3498db' ) );
    $card_bg = sanitize_hex_color( get_theme_mod( 'card_background_color', '#ffffff' ) );
    $card_radius = absint( get_theme_mod( 'card_border_radius', '15' ) );
    $dot_size = absint( get_theme_mod( 'timeline_dot_size', '20' ) );
    $site_title_size = chronocards_ruchir_sanitize_font_size( get_theme_mod( 'site_title_font_size', '2.5' ) );
    $card_title_size = chronocards_ruchir_sanitize_font_size( get_theme_mod( 'card_title_font_size', '1.5' ) );
    $container_width = absint( get_theme_mod( 'container_max_width', '1200' ) );
    $card_spacing = absint( get_theme_mod( 'card_spacing', '40' ) );
    
    // Ensure fallback values if sanitization fails.
    if ( empty( $timeline_color ) ) {
        $timeline_color = '#3498db';
    }
    if ( empty( $card_bg ) ) {
        $card_bg = '#ffffff';
    }
    if ( empty( $card_radius ) ) {
        $card_radius = 15;
    }
    if ( empty( $dot_size ) ) {
        $dot_size = 20;
    }
    if ( empty( $site_title_size ) ) {
        $site_title_size = '2.5';
    }
    if ( empty( $card_title_size ) ) {
        $card_title_size = '1.5';
    }
    if ( empty( $container_width ) ) {
        $container_width = 1200;
    }
    if ( empty( $card_spacing ) ) {
        $card_spacing = 40;
    }
    
    // Values are already sanitized via sanitize_callback in get_theme_mod()
    // Ensure numeric values are properly formatted
    $card_radius = absint( $card_radius );
    $dot_size = absint( $dot_size );
    $container_width = absint( $container_width );
    $card_spacing = absint( $card_spacing );
    
    // Font sizes are already sanitized via chronocards_ruchir_sanitize_font_size()
    // Colors are already sanitized via sanitize_hex_color()
    // Output CSS directly without escaping (values are already sanitized)
    $css = "
        .timeline-line {
            background: linear-gradient(to bottom, {$timeline_color}, #e74c3c, #2ecc71, #f39c12);
        }
        .timeline-card {
            background-color: {$card_bg};
            border-radius: {$card_radius}px;
            margin: {$card_spacing}px 0;
        }
        .timeline-dot {
            width: {$dot_size}px;
            height: {$dot_size}px;
        }
        .site-title {
            font-size: {$site_title_size}rem;
        }
        .card-title {
            font-size: {$card_title_size}rem;
        }
        .container {
            max-width: {$container_width}px;
        }
    ";
    
    wp_add_inline_style( 'chronocards-ruchir-style', $css );
}
add_action( 'wp_enqueue_scripts', 'chronocards_ruchir_customizer_css' );
