<?php
/**
 * ChronoCards Ruchir functions and definitions
 *
 * @package ChronoCards_Ruchir
 */

// Prevent direct access.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 */
function chronocards_ruchir_setup() {
	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	// Let WordPress manage the document title.
	add_theme_support( 'title-tag' );

	// Enable support for Post Thumbnails on posts and pages.
	add_theme_support( 'post-thumbnails' );

	// Set custom image sizes.
	add_image_size( 'chronocards-featured', 400, 300, true );
	add_image_size( 'chronocards-large', 800, 600, true );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus(
		array(
			'primary' => esc_html__( 'Primary Menu', 'chronocards-ruchir' ),
		)
	);

	// Switch default core markup for search form, comment form, and comments.
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'style',
			'script',
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	// Add support for core custom logo.
	add_theme_support(
		'custom-logo',
		array(
			'height'      => 250,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => true,
		)
	);

	// Add support for custom colors.
	add_theme_support(
		'custom-background',
		array(
			'default-color' => 'ffffff',
		)
	);

	// Add support for editor styles.
	add_theme_support( 'editor-styles' );
	add_editor_style( 'editor-style.css' );

	// Add support for responsive embeds.
	add_theme_support( 'responsive-embeds' );

	// Add support for wide and full alignment.
	add_theme_support( 'align-wide' );

	// Add support for custom line height.
	add_theme_support( 'custom-line-height' );

	// Add support for custom units.
	add_theme_support( 'custom-units' );

	// Add support for custom spacing.
	add_theme_support( 'custom-spacing' );

	// Add support for wp-block-styles.
	add_theme_support( 'wp-block-styles' );
}
add_action( 'after_setup_theme', 'chronocards_ruchir_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 */
function chronocards_ruchir_content_width() {
	// phpcs:disable WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
	$GLOBALS['content_width'] = apply_filters( 'chronocards_ruchir_content_width', 1200 );
	// phpcs:enable
}
add_action( 'after_setup_theme', 'chronocards_ruchir_content_width', 0 );

/**
 * Register widget areas.
 */
function chronocards_ruchir_widgets_init() {
	register_sidebar(
		array(
			'name'          => esc_html__( 'Sidebar', 'chronocards-ruchir' ),
			'id'            => 'sidebar-1',
			'description'   => esc_html__( 'Add widgets here.', 'chronocards-ruchir' ),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);

	register_sidebar(
		array(
			'name'          => esc_html__( 'Footer Widgets', 'chronocards-ruchir' ),
			'id'            => 'footer-1',
			'description'   => esc_html__( 'Add footer widgets here.', 'chronocards-ruchir' ),
			'before_widget' => '<div id="%1$s" class="footer-widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h3 class="footer-widget-title">',
			'after_title'   => '</h3>',
		)
	);
}
add_action( 'widgets_init', 'chronocards_ruchir_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function chronocards_ruchir_scripts() {
	// Enqueue main stylesheet.
	wp_enqueue_style( 'chronocards-ruchir-style', get_stylesheet_uri(), array(), wp_get_theme()->get( 'Version' ) );

	// Enqueue custom JavaScript.
	wp_enqueue_script( 'chronocards-ruchir-script', get_template_directory_uri() . '/js/chronocards.js', array( 'jquery' ), wp_get_theme()->get( 'Version' ), true );

	// Localize script for AJAX.
	wp_localize_script(
		'chronocards-ruchir-script',
		'chronocards_ajax',
		array(
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce'    => wp_create_nonce( 'chronocards_nonce' ),
		)
	);

	// Enqueue comment reply script.
	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'chronocards_ruchir_scripts' );

/**
 * Custom excerpt length
 */
function chronocards_ruchir_excerpt_length() {
	return 25;
}
add_filter( 'excerpt_length', 'chronocards_ruchir_excerpt_length', 999 );

/**
 * Custom excerpt more
 *
 * @param string $more The string shown within the more link.
 * @return string Custom excerpt more string.
 */
function chronocards_ruchir_excerpt_more( $more ) {
	if ( is_admin() ) {
		return $more;
	}
	return '...';
}
add_filter( 'excerpt_more', 'chronocards_ruchir_excerpt_more' );

/**
 * Add custom body classes
 *
 * @param array $classes Array of CSS classes for the body element.
 * @return array Modified array of CSS classes.
 */
function chronocards_ruchir_body_classes( $classes ) {
	// Add class for timeline layout.
	if ( is_home() || is_archive() ) {
		$classes[] = 'timeline-layout';
	}

	// Add class for single post.
	if ( is_single() ) {
		$classes[] = 'single-post-layout';
	}

	return $classes;
}
add_filter( 'body_class', 'chronocards_ruchir_body_classes' );

/**
 * Customize the read more link
 */
function chronocards_ruchir_read_more_link() {
	return '<a class="read-more-link" href="' . get_permalink() . '">' . __( 'Read More', 'chronocards-ruchir' ) . '</a>';
}
add_filter( 'the_content_more_link', 'chronocards_ruchir_read_more_link' );

/**
 * Add custom colors for categories
 */
function chronocards_ruchir_category_colors() {
	$colors = array(
		'tech'      => '#3498db',
		'lifestyle' => '#2ecc71',
		'business'  => '#e74c3c',
		'design'    => '#f39c12',
		'travel'    => '#9b59b6',
		'food'      => '#e67e22',
		'health'    => '#1abc9c',
		'education' => '#34495e',
	);

	return apply_filters( 'chronocards_ruchir_category_colors', $colors );
}

/**
 * Get category color
 *
 * @param string $category_slug The slug of the category.
 * @return string The color code for the category.
 */
function chronocards_ruchir_get_category_color( $category_slug ) {
	$colors = chronocards_ruchir_category_colors();
	return isset( $colors[ $category_slug ] ) ? $colors[ $category_slug ] : '#3498db';
}

/**
 * Add custom CSS for category colors
 */
function chronocards_ruchir_category_css() {
	$colors = chronocards_ruchir_category_colors();
	$css    = '';

	foreach ( $colors as $category => $color ) {
		$category_slug = sanitize_html_class( $category );
		$color_hex     = sanitize_hex_color( $color );
		if ( empty( $color_hex ) ) {
			$color_hex = '#3498db'; // Default fallback.
		}
		$css .= ".category-{$category_slug} .timeline-dot,
                 .category-{$category_slug} .card-category {
                     background-color: {$color_hex} !important;
                 }";
	}

	wp_add_inline_style( 'chronocards-ruchir-style', wp_strip_all_tags( $css ) );
}
add_action( 'wp_enqueue_scripts', 'chronocards_ruchir_category_css' );

/**
 * AJAX handler for expanding cards
 */
function chronocards_ruchir_expand_card() {
	// Verify nonce.
	if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['nonce'] ) ), 'chronocards_nonce' ) ) {
		wp_die( 'Security check failed' );
	}

	$post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
	$post    = get_post( $post_id );

	if ( ! $post ) {
		wp_die( 'Post not found' );
	}

	setup_postdata( $post );

	ob_start();
	?>
	<div class="entry-content">
		<?php
		the_content();

		wp_link_pages(
			array(
				'before' => '<div class="page-links">' . esc_html__( 'Pages:', 'chronocards-ruchir' ),
				'after'  => '</div>',
			)
		);
		?>
	</div>
	<?php
	$content = ob_get_clean();

	wp_reset_postdata();

	wp_send_json_success(
		array(
			'content' => $content,
		)
	);
}
add_action( 'wp_ajax_expand_card', 'chronocards_ruchir_expand_card' );
add_action( 'wp_ajax_nopriv_expand_card', 'chronocards_ruchir_expand_card' );


/**
 * Add theme support for block editor
 */
function chronocards_ruchir_block_editor_styles() {
	wp_enqueue_style( 'chronocards-ruchir-block-editor-style', get_template_directory_uri() . '/editor-style.css', array(), wp_get_theme()->get( 'Version' ) );
}
add_action( 'enqueue_block_editor_assets', 'chronocards_ruchir_block_editor_styles' );

/**
 * Register block styles for theme blocks
 */
function chronocards_ruchir_register_block_styles() {
	// Register timeline text style for paragraph block
	register_block_style(
		'core/paragraph',
		array(
			'name'         => 'timeline-text',
			'label'        => esc_html__( 'Timeline Text', 'chronocards-ruchir' ),
		)
	);

	// Register timeline card style for group block
	register_block_style(
		'core/group',
		array(
			'name'         => 'timeline-card',
			'label'        => esc_html__( 'Timeline Card', 'chronocards-ruchir' ),
		)
	);

	// Register featured image style for image block
	register_block_style(
		'core/image',
		array(
			'name'         => 'featured-image',
			'label'        => esc_html__( 'Featured Image', 'chronocards-ruchir' ),
		)
	);
}
add_action( 'init', 'chronocards_ruchir_register_block_styles' );

/**
 * Register block pattern category
 */
function chronocards_ruchir_register_block_pattern_category() {
	register_block_pattern_category(
		'chronocards-ruchir',
		array(
			'label' => esc_html__( 'ChronoCards Ruchir', 'chronocards-ruchir' ),
		)
	);
}
add_action( 'init', 'chronocards_ruchir_register_block_pattern_category' );

/**
 * Register block patterns
 */
function chronocards_ruchir_register_block_patterns() {
	// Timeline Card Pattern
	register_block_pattern(
		'chronocards-ruchir/timeline-card',
		array(
			'title'       => esc_html__( 'Timeline Card', 'chronocards-ruchir' ),
			'description' => esc_html__( 'A card layout perfect for timeline posts with featured image and content.', 'chronocards-ruchir' ),
			'categories'  => array( 'chronocards-ruchir' ),
			'content'     => '<!-- wp:group {"className":"is-style-timeline-card","layout":{"type":"constrained"}} -->
<div class="wp-block-group is-style-timeline-card">
	<!-- wp:heading {"level":3} -->
	<h3 class="wp-block-heading">' . esc_html__( 'Card Title', 'chronocards-ruchir' ) . '</h3>
	<!-- /wp:heading -->
	
	<!-- wp:image {"className":"is-style-featured-image"} -->
	<figure class="wp-block-image is-style-featured-image"><img alt=""/></figure>
	<!-- /wp:image -->
	
	<!-- wp:paragraph {"className":"is-style-timeline-text"} -->
	<p class="is-style-timeline-text">' . esc_html__( 'Add your content here. This card style is perfect for timeline posts and milestones.', 'chronocards-ruchir' ) . '</p>
	<!-- /wp:paragraph -->
</div>
<!-- /wp:group -->',
		)
	);

	// Hero Section Pattern
	register_block_pattern(
		'chronocards-ruchir/hero-section',
		array(
			'title'       => esc_html__( 'Hero Section', 'chronocards-ruchir' ),
			'description' => esc_html__( 'A hero section with title, description, and call-to-action button.', 'chronocards-ruchir' ),
			'categories'  => array( 'chronocards-ruchir' ),
			'content'     => '<!-- wp:group {"className":"hero-section","layout":{"type":"constrained"}} -->
<div class="wp-block-group hero-section">
	<!-- wp:heading {"textAlign":"center","level":1} -->
	<h1 class="wp-block-heading has-text-align-center">' . esc_html__( 'Welcome to ChronoCards', 'chronocards-ruchir' ) . '</h1>
	<!-- /wp:heading -->
	
	<!-- wp:paragraph {"align":"center"} -->
	<p class="has-text-align-center">' . esc_html__( 'A modern storytelling theme that combines timeline layouts with card-based designs.', 'chronocards-ruchir' ) . '</p>
	<!-- /wp:paragraph -->
	
	<!-- wp:buttons {"layout":{"type":"flex","justifyContent":"center"}} -->
	<div class="wp-block-buttons">
		<!-- wp:button -->
		<div class="wp-block-button"><a class="wp-block-button__link wp-element-button">' . esc_html__( 'Get Started', 'chronocards-ruchir' ) . '</a></div>
		<!-- /wp:button -->
	</div>
	<!-- /wp:buttons -->
</div>
<!-- /wp:group -->',
		)
	);

	// Two Column Content Pattern
	register_block_pattern(
		'chronocards-ruchir/two-column-content',
		array(
			'title'       => esc_html__( 'Two Column Content', 'chronocards-ruchir' ),
			'description' => esc_html__( 'A two-column layout with image and text content.', 'chronocards-ruchir' ),
			'categories'  => array( 'chronocards-ruchir' ),
			'content'     => '<!-- wp:columns -->
<div class="wp-block-columns">
	<!-- wp:column -->
	<div class="wp-block-column">
		<!-- wp:image {"className":"is-style-featured-image"} -->
		<figure class="wp-block-image is-style-featured-image"><img alt=""/></figure>
		<!-- /wp:image -->
	</div>
	<!-- /wp:column -->
	
	<!-- wp:column -->
	<div class="wp-block-column">
		<!-- wp:heading {"level":3} -->
		<h3 class="wp-block-heading">' . esc_html__( 'Content Title', 'chronocards-ruchir' ) . '</h3>
		<!-- /wp:heading -->
		
		<!-- wp:paragraph {"className":"is-style-timeline-text"} -->
		<p class="is-style-timeline-text">' . esc_html__( 'Add your content here. This two-column layout is perfect for showcasing images alongside text content.', 'chronocards-ruchir' ) . '</p>
		<!-- /wp:paragraph -->
	</div>
	<!-- /wp:column -->
</div>
<!-- /wp:columns -->',
		)
	);
}
add_action( 'init', 'chronocards_ruchir_register_block_patterns' );


/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Custom Walker for Navigation Menu.
 */
require get_template_directory() . '/inc/class-chronocards-walker-nav-menu.php';



