<?php

namespace Codemanas\ThemeInfo;

use WP_Error;

class Notices {

	private static ?Notices $_instance = null;

	public static function get_instance(): ?Notices {
		return is_null( self::$_instance ) ? self::$_instance = new self() : self::$_instance;
	}

	function is_plugin_installed( $plugin_slug ) {
		// Ensure the get_plugins() function is available
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		// Get the list of all installed plugins
		$installed_plugins = get_plugins();

		// Check if the plugin exists in the installed plugins list
		return isset( $installed_plugins[ $plugin_slug ] );
	}


	private function is_cm_blocks_installed(): bool {
		return $this->is_plugin_installed( 'cm-blocks/cm-blocks.php' );
	}

	private function is_cm_blocks_active(): bool {
		return defined( 'CM_BLOCK_VERSION' );
	}

	protected function __construct() {
		add_action( 'admin_notices', [ $this, 'printNotice' ] );
		add_action( 'wp_ajax_cm_theme_dismiss_notice', [ $this, 'handle_dismiss' ] );
		add_action( 'wp_ajax_cm_theme_install_cm_blocks', [ $this, 'handle_install_cm_blocks' ] );
	}

	public function handle_dismiss() {
		$nonce = sanitize_text_field( filter_input( INPUT_POST, 'nonce' ) );
		if ( ! wp_verify_nonce( $nonce, 'cm-theme-dismiss-notice' ) ) {
			wp_send_json_error( [ 'Invalid nonce', $nonce ] );
		}
		$user = get_current_user_id();
		update_user_meta( $user, 'cm_theme_user_dismissed_notice', true );
		wp_send_json_success( 'Success' );
	}

	public function handle_install_cm_blocks() {
		$nonce = sanitize_text_field( filter_input( INPUT_POST, 'nonce' ) );
		if ( ! wp_verify_nonce( $nonce, 'cm-theme-install-cm-blocks' ) ) {
			wp_send_json_error( [ 'Invalid nonce', $nonce ] );
		}

		$installer = new PluginInstaller( 'cm-blocks', 'cm-blocks' );
		$result    = $installer->installAndActivate();
		if ( $result instanceof WP_Error ) {
			if ( is_wp_error( $result ) ) {
				wp_send_json_error( array( 'message' => $result->get_error_message() ) );
			}
		}

		wp_send_json_success( array( 'message' => $result ) );
	}

	public function printNotice() {
		if ( ! current_user_can( 'install_plugins' ) ) {
			return;
		}


		global $pagenow;
		$user_id          = get_current_user_id();
		$notice_dismissed = get_user_meta( $user_id, 'cm_theme_user_dismissed_notice', true );
		$page             = filter_input( INPUT_GET, 'page', FILTER_UNSAFE_RAW );
		if ( defined( 'CM_BLOCK_VERSION' ) ) {
			return;
		} elseif ( ! empty( $notice_dismissed ) ) {
			return;
		} elseif ( isset( $pagenow ) && $pagenow == 'admin.php' && $page == 'canvas-info' ) {
			return;
		}
		?>
        <style>
            .cm-theme-notice {
                padding: 25px 20px;
            }

            .cm-theme-notice-wrapper {
                display: flex;
                flex-wrap: wrap;
                gap: 20px;
            }

            .cm-theme-notice__logo {
                width: 40px;
                height: 40px;
            }

            .cm-theme-notice__copy {
                flex-shrink: 1;
                max-width: 800px;
            }

            .cm-theme-notice-heading {
                margin: 0;
                font-weight: normal;
                font-size: 20px;
            }

            .cm-theme-notice__copy-actions {
                display: flex;
                align-items: center;
                gap: 20px;
                margin-top: 20px;
            }
        </style>
        <script>
          (function () {
            var CMThemeBlocksNotice = () => {
              const DOM = {
                container: null,
                dismissButton: null,
                installButton: null,
              }

              function observeDismissButton () {
                const observer = new MutationObserver((mutations) => {
                  mutations.forEach((mutation) => {
                    if (mutation.type === 'childList') {
                      const dismissButton = DOM.container.querySelector('.notice-dismiss')
                      if (dismissButton) {
                        dismissButton.addEventListener('click', handleClosingNotice)
                        observer.disconnect() // Stop observing once the button is found

                      }
                    }
                  })
                })

                observer.observe(DOM.container, { childList: true, subtree: true })
              }

              function cacheDOM (container) {
                DOM.container = container
                DOM.installButton = container.querySelector('.cm-theme-install-cm-blocks')
              }

              function handleClosingNotice () {
                const formData = new FormData()
                formData.append('action', 'cm_theme_dismiss_notice')
                formData.append('nonce', '<?php echo esc_js( wp_create_nonce( 'cm-theme-dismiss-notice' ) ); ?>')

                // Use Fetch API
                fetch(ajaxurl, {
                  method: 'POST',
                  body: formData,
                }).then((response) => response.json()) // Parse JSON response
                  .then((data) => {
                  }).catch((error) => {
                  console.error('AJAX error:', error)
                })
              }

              function handlePluginInstall (e) {
                e.preventDefault()
                const formData = new FormData()
                formData.append('action', 'cm_theme_install_cm_blocks')
                formData.append('nonce', '<?php echo esc_js( wp_create_nonce( 'cm-theme-install-cm-blocks' ) ); ?>')
                //DOM.installButton.text =
                DOM.installButton.disabled = true
                const basePath = '/wp-admin/admin.php'
                const params = new URLSearchParams({
                  page: 'cm-blocks',
                  path: 'starter-sites',
                })
                const fullURL = `${window.location.origin}${basePath}?${params.toString()}`

                // Use Fetch API
                fetch(ajaxurl, {
                  method: 'POST',
                  body: formData,
                }).then((response) => response.json()) // Parse JSON response
                  .then((data) => {
                    if (data.success) {
                      // Redirect to the constructed URL
                      window.location.href = fullURL
                    }
                  }).catch((error) => {
                  console.error('AJAX error:', error)
                })
              }

              function setupEventListeners () {
                if (DOM.installButton !== null) DOM.installButton.addEventListener('click', handlePluginInstall)
              }

              function init () {
                const container = document.querySelector('.cm-theme-notice')
                if (container === null) return
                cacheDOM(container)
                observeDismissButton()
                setupEventListeners()
              }

              return { init }
            }

            document.addEventListener('DOMContentLoaded', () => {
              CMThemeBlocksNotice().init()
            })

          })()
        </script>
		<?php
        $theme = wp_get_theme();
		$text = Translations::get_instance()->get_notice_translations();
		?>
        <div class="cm-theme-notice notice is-dismissible notice-info">
            <div class="cm-theme-notice-wrapper">
                <img src="<?php echo esc_url( get_template_directory_uri() . '/vendor/codemanas/theme-info/img/cm-block-logo.webp' ); ?>" class="cm-theme-notice__logo" alt="CM Blocks"/>
                <div class="cm-theme-notice__copy">
                    <h2 class="cm-theme-notice-heading"><?php echo esc_html( sprintf( $text['title'], $theme->get('Name') ) ) ?></h2>
                    <p>We strongly recommend you to use the <b>CM Blocks</b> plugin.<br>Simple to use and designed for impact, CM Blocks makes creating stunning, user-friendly sites a breeze.<br/>Unlock powerful features like starter sites, a design library, and versatile blocks for endless creativity!</p>
                    <div class="cm-theme-notice__copy-actions">
                        <?php PluginInstaller::render_installer_button('cm-blocks','cm-blocks', ['cm-theme-install-cm-blocks']); ?>
                        <a href="https://cmblocks.com" target="_blank">Why use CM Blocks ?</a>
                    </div>
                </div>
            </div>
        </div>
		<?php
	}
}